import {addClass} from './../../helpers/dom/element';

/**
 * Comment editor for the Comments plugin.
 *
 * @class CommentEditor
 * @plugin Comments
 */
class CommentEditor {
  static get CLASS_EDITOR_CONTAINER() {
    return 'htCommentsContainer';
  }

  static get CLASS_EDITOR() {
    return 'htComments';
  }

  static get CLASS_INPUT() {
    return 'htCommentTextArea';
  }

  static get CLASS_CELL() {
    return 'htCommentCell';
  }

  constructor() {
    this.editor = this.createEditor();
    this.editorStyle = this.editor.style;

    this.hidden = true;

    this.hide();
  }

  /**
   * Set position of the comments editor according to the  provided x and y coordinates.
   *
   * @param {Number} x X position (in pixels).
   * @param {Number} y Y position (in pixels).
   */
  setPosition(x, y) {
    this.editorStyle.left = x + 'px';
    this.editorStyle.top = y + 'px';
  }

  /**
   * Set the editor size according to the provided arguments.
   *
   * @param {Number} width Width in pixels.
   * @param {Number} height Height in pixels.
   */
  setSize(width, height) {
    if (width && height) {
      const input = this.getInputElement();

      input.style.width = width + 'px';
      input.style.height = height + 'px';
    }
  }

  /**
   * Reset the editor size to its initial state.
   */
  resetSize() {
    const input = this.getInputElement();

    input.style.width = '';
    input.style.height = '';
  }

  /**
   * Set the read-only state for the comments editor.
   *
   * @param {Boolean} state The new read only state.
   */
  setReadOnlyState(state) {
    const input = this.getInputElement();

    input.readOnly = state;
  }

  /**
   * Show the comments editor.
   */
  show() {
    this.editorStyle.display = 'block';
    this.hidden = false;
  }

  /**
   * Hide the comments editor.
   */
  hide() {
    this.editorStyle.display = 'none';
    this.hidden = true;
  }

  /**
   * Checks if the editor is visible.
   *
   * @returns {Boolean}
   */
  isVisible() {
    return this.editorStyle.display === 'block';
  }

  /**
   * Set the comment value.
   *
   * @param {String} [value] The value to use.
   */
  setValue(value = '') {
    value = value || '';
    this.getInputElement().value = value;
  }

  /**
   * Get the comment value.
   *
   * @returns {String}
   */
  getValue() {
    return this.getInputElement().value;
  }

  /**
   * Checks if the comment input element is focused.
   *
   * @returns {Boolean}
   */
  isFocused() {
    return document.activeElement === this.getInputElement();
  }

  /**
   * Focus the comments input element.
   */
  focus() {
    this.getInputElement().focus();
  }

  /**
   * Create the `textarea` to be used as a comments editor.
   *
   * @returns {HTMLElement}
   */
  createEditor() {
    let container = document.querySelector('.' + CommentEditor.CLASS_EDITOR_CONTAINER);
    let editor;
    let textArea;

    if (!container) {
      container = document.createElement('div');
      addClass(container, CommentEditor.CLASS_EDITOR_CONTAINER);
      document.body.appendChild(container);
    }
    editor = document.createElement('div');
    addClass(editor, CommentEditor.CLASS_EDITOR);

    textArea = document.createElement('textarea');
    addClass(textArea, CommentEditor.CLASS_INPUT);

    editor.appendChild(textArea);
    container.appendChild(editor);

    return editor;
  }

  /**
   * Get the input element.
   *
   * @returns {HTMLElement}
   */
  getInputElement() {
    return this.editor.querySelector('.' + CommentEditor.CLASS_INPUT);
  }

  /**
   * Destroy the comments editor.
   */
  destroy() {
    this.editor.parentNode.removeChild(this.editor);
    this.editor = null;
    this.editorStyle = null;
  }
}

export {CommentEditor};
