"use strict";

exports.__esModule = true;
exports.compareFunctionFactory = compareFunctionFactory;
var _mixed = require("../../../helpers/mixed");
var _sortService = require("../sortService");
/**
 * Default sorting compare function factory. Method get as parameters `sortOrder` and `columnMeta` and return compare function.
 *
 * @param {string} sortOrder Sort order (`asc` for ascending, `desc` for descending).
 * @param {object} columnMeta Column meta object.
 * @param {object} columnPluginSettings Plugin settings for the column.
 * @returns {Function} The compare function.
 */
function compareFunctionFactory(sortOrder, columnMeta, columnPluginSettings) {
  const locale = columnMeta.locale;
  return function (value, nextValue) {
    const {
      sortEmptyCells
    } = columnPluginSettings;
    if (typeof value === 'string') {
      value = value.toLocaleLowerCase(locale);
    }
    if (typeof nextValue === 'string') {
      nextValue = nextValue.toLocaleLowerCase(locale);
    }
    if (value === nextValue) {
      return _sortService.DO_NOT_SWAP;
    }
    if ((0, _mixed.isEmpty)(value)) {
      if ((0, _mixed.isEmpty)(nextValue)) {
        return _sortService.DO_NOT_SWAP;
      }

      // Just fist value is empty and `sortEmptyCells` option was set
      if (sortEmptyCells) {
        return sortOrder === 'asc' ? _sortService.FIRST_BEFORE_SECOND : _sortService.FIRST_AFTER_SECOND;
      }
      return _sortService.FIRST_AFTER_SECOND;
    }
    if ((0, _mixed.isEmpty)(nextValue)) {
      // Just second value is empty and `sortEmptyCells` option was set
      if (sortEmptyCells) {
        return sortOrder === 'asc' ? _sortService.FIRST_AFTER_SECOND : _sortService.FIRST_BEFORE_SECOND;
      }
      return _sortService.FIRST_BEFORE_SECOND;
    }
    if (isNaN(value) && !isNaN(nextValue)) {
      return sortOrder === 'asc' ? _sortService.FIRST_AFTER_SECOND : _sortService.FIRST_BEFORE_SECOND;
    } else if (!isNaN(value) && isNaN(nextValue)) {
      return sortOrder === 'asc' ? _sortService.FIRST_BEFORE_SECOND : _sortService.FIRST_AFTER_SECOND;
    } else if (!(isNaN(value) || isNaN(nextValue))) {
      value = parseFloat(value);
      nextValue = parseFloat(nextValue);
    }
    if (value < nextValue) {
      return sortOrder === 'asc' ? _sortService.FIRST_BEFORE_SECOND : _sortService.FIRST_AFTER_SECOND;
    }
    if (value > nextValue) {
      return sortOrder === 'asc' ? _sortService.FIRST_AFTER_SECOND : _sortService.FIRST_BEFORE_SECOND;
    }
    return _sortService.DO_NOT_SWAP;
  };
}
const COLUMN_DATA_TYPE = exports.COLUMN_DATA_TYPE = 'default';