"use strict";

exports.__esModule = true;
exports.getRegisteredMapsCounter = getRegisteredMapsCounter;
require("core-js/modules/es.error.cause.js");
require("core-js/modules/esnext.iterator.constructor.js");
require("core-js/modules/esnext.iterator.for-each.js");
var _mixed = require("../../helpers/mixed");
var _object = require("../../helpers/object");
var _localHooks = _interopRequireDefault(require("../../mixins/localHooks"));
function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
function _defineProperty(e, r, t) { return (r = _toPropertyKey(r)) in e ? Object.defineProperty(e, r, { value: t, enumerable: !0, configurable: !0, writable: !0 }) : e[r] = t, e; }
function _toPropertyKey(t) { var i = _toPrimitive(t, "string"); return "symbol" == typeof i ? i : i + ""; }
function _toPrimitive(t, r) { if ("object" != typeof t || !t) return t; var e = t[Symbol.toPrimitive]; if (void 0 !== e) { var i = e.call(t, r || "default"); if ("object" != typeof i) return i; throw new TypeError("@@toPrimitive must return a primitive value."); } return ("string" === r ? String : Number)(t); }
// Counter for checking if there is a memory leak.
let registeredMaps = 0;

/**
 * Collection of index maps having unique names. It allow us to perform bulk operations such as init, remove, insert on all index maps that have been registered in the collection.
 */
class MapCollection {
  constructor() {
    /**
     * Collection of index maps.
     *
     * @type {Map<string, IndexMap>}
     */
    _defineProperty(this, "collection", new Map());
  }
  /**
   * Register custom index map.
   *
   * @param {string} uniqueName Unique name of the index map.
   * @param {IndexMap} indexMap Index map containing miscellaneous (i.e. Meta data, indexes sequence), updated after remove and insert data actions.
   */
  register(uniqueName, indexMap) {
    if (this.collection.has(uniqueName) === false) {
      this.collection.set(uniqueName, indexMap);
      indexMap.addLocalHook('change', () => this.runLocalHooks('change', indexMap));
      registeredMaps += 1;
    }
  }

  /**
   * Unregister custom index map.
   *
   * @param {string} name Name of the index map.
   */
  unregister(name) {
    const indexMap = this.collection.get(name);
    if ((0, _mixed.isDefined)(indexMap)) {
      indexMap.destroy();
      this.collection.delete(name);
      this.runLocalHooks('change', indexMap);
      registeredMaps -= 1;
    }
  }

  /**
   * Unregisters and destroys all collected index map instances.
   */
  unregisterAll() {
    this.collection.forEach((indexMap, name) => this.unregister(name));
    this.collection.clear();
  }

  /**
   * Get index map for the provided name.
   *
   * @param {string} [name] Name of the index map.
   * @returns {Array|IndexMap}
   */
  get(name) {
    if ((0, _mixed.isUndefined)(name)) {
      return Array.from(this.collection.values());
    }
    return this.collection.get(name);
  }

  /**
   * Get collection size.
   *
   * @returns {number}
   */
  getLength() {
    return this.collection.size;
  }

  /**
   * Remove some indexes and corresponding mappings and update values of the others within all collection's index maps.
   *
   * @private
   * @param {Array} removedIndexes List of removed indexes.
   */
  removeFromEvery(removedIndexes) {
    this.collection.forEach(indexMap => {
      indexMap.remove(removedIndexes);
    });
  }

  /**
   * Insert new indexes and corresponding mapping and update values of the others all collection's index maps.
   *
   * @private
   * @param {number} insertionIndex Position inside the actual list.
   * @param {Array} insertedIndexes List of inserted indexes.
   */
  insertToEvery(insertionIndex, insertedIndexes) {
    this.collection.forEach(indexMap => {
      indexMap.insert(insertionIndex, insertedIndexes);
    });
  }

  /**
   * Set default values to index maps within collection.
   *
   * @param {number} length Destination length for all stored maps.
   */
  initEvery(length) {
    this.collection.forEach(indexMap => {
      indexMap.init(length);
    });
  }
}
exports.MapCollection = MapCollection;
(0, _object.mixin)(MapCollection, _localHooks.default);

/**
 * @returns {number}
 */
function getRegisteredMapsCounter() {
  return registeredMaps;
}