/*!
 * Pikaday
 *
 * Copyright © 2014 David Bushell | BSD & MIT license | https://github.com/Pikaday/Pikaday
 */
const getWindow = () => window;
const getDocument = () => window.document;
const sto = (fn, delay) => window.setTimeout(fn, delay);
const addEvent = function(el, e, callback, capture) {
  el.addEventListener(e, callback, !!capture);
};
const removeEvent = function(el, e, callback, capture) {
  el.removeEventListener(e, callback, !!capture);
};
const trim = function(str) {
  return str.trim ? str.trim() : str.replace(/^\s+|\s+$/g, "");
};
const hasClass = function(el, cn) {
  return (" " + el.className + " ").indexOf(" " + cn + " ") !== -1;
};
const addClass = function(el, cn) {
  if (!hasClass(el, cn)) {
    el.className = el.className === "" ? cn : el.className + " " + cn;
  }
};
const removeClass = function(el, cn) {
  el.className = trim((" " + el.className + " ").replace(" " + cn + " ", " "));
};
const isArray = function(obj) {
  return /Array/.test(Object.prototype.toString.call(obj));
};
const isDate = function(obj) {
  return /Date/.test(Object.prototype.toString.call(obj)) && !isNaN(obj.getTime());
};
const isWeekend = function(date) {
  let day = date.getDay();
  return day === 0 || day === 6;
};
const isLeapYear = function(year) {
  return year % 4 === 0 && year % 100 !== 0 || year % 400 === 0;
};
const getDaysInMonth = function(year, month) {
  return [31, isLeapYear(year) ? 29 : 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31][month];
};
const setToStartOfDay = function(date) {
  if (isDate(date)) {
    date.setHours(0, 0, 0, 0);
  }
};
const compareDates = function(a, b) {
  return a.getTime() === b.getTime();
};
const extend = function(to, from, overwrite) {
  let prop, hasProp;
  for (prop in from) {
    hasProp = to[prop] !== void 0;
    if (hasProp && typeof from[prop] === "object" && from[prop] !== null && from[prop].nodeName === void 0) {
      if (isDate(from[prop])) {
        if (overwrite) {
          to[prop] = new Date(from[prop].getTime());
        }
      } else if (isArray(from[prop])) {
        if (overwrite) {
          to[prop] = from[prop].slice(0);
        }
      } else {
        to[prop] = extend({}, from[prop], overwrite);
      }
    } else if (overwrite || !hasProp) {
      to[prop] = from[prop];
    }
  }
  return to;
};
const fireEvent = function(el, eventName, data) {
  let ev = getDocument().createEvent("HTMLEvents");
  ev.initEvent(eventName, true, false);
  ev = extend(ev, data);
  el.dispatchEvent(ev);
};
const adjustCalendar = function(calendar) {
  if (calendar.month < 0) {
    calendar.year -= Math.ceil(Math.abs(calendar.month) / 12);
    calendar.month += 12;
  }
  if (calendar.month > 11) {
    calendar.year += Math.floor(Math.abs(calendar.month) / 12);
    calendar.month -= 12;
  }
  return calendar;
};
const defaults = {
  // bind the picker to a form field
  field: null,
  // automatically show/hide the picker on `field` focus (default `true` if `field` is set)
  bound: void 0,
  // data-attribute on the input field with an aria assistance text (only applied when `bound` is set)
  ariaLabel: "Use the arrow keys to pick a date",
  // position of the datepicker, relative to the field (default to bottom & left)
  // ('bottom' & 'left' keywords are not used, 'top' & 'right' are modifier on the bottom/left position)
  position: "bottom left",
  // automatically fit in the viewport even if it means repositioning from the position option
  reposition: true,
  // the default output format for `.toString()` and `field` value
  format: "YYYY-MM-DD",
  // the toString function which gets passed a current date object and format
  // and returns a string
  toString: null,
  // used to create date object from current input string
  parse: null,
  // the initial date to view when first opened
  defaultDate: null,
  // make the `defaultDate` the initial selected value
  setDefaultDate: false,
  // first day of week (0: Sunday, 1: Monday etc)
  firstDay: 0,
  // minimum number of days in the week that gets week number one
  // default ISO 8601, week 01 is the week with the first Thursday (4)
  firstWeekOfYearMinDays: 4,
  // the default flag for moment's strict date parsing
  formatStrict: false,
  // the minimum/earliest date that can be selected
  minDate: null,
  // the maximum/latest date that can be selected
  maxDate: null,
  // number of years either side, or array of upper/lower range
  yearRange: 10,
  // show week numbers at head of row
  showWeekNumber: false,
  // Week picker mode
  pickWholeWeek: false,
  // used internally (don't config outside)
  minYear: 0,
  maxYear: 9999,
  minMonth: void 0,
  maxMonth: void 0,
  startRange: null,
  endRange: null,
  isRTL: false,
  // Additional text to append to the year in the calendar title
  yearSuffix: "",
  // Render the month after year in the calendar title
  showMonthAfterYear: false,
  // Render days of the calendar grid that fall in the next or previous month
  showDaysInNextAndPreviousMonths: false,
  // Allows user to select days that fall in the next or previous month
  enableSelectionDaysInNextAndPreviousMonths: false,
  // how many months are visible
  numberOfMonths: 1,
  // when numberOfMonths is used, this will help you to choose where the main calendar will be (default `left`, can be set to `right`)
  // only used for the first display or when a selected date is not visible
  mainCalendar: "left",
  // Specify a DOM element to render the calendar in
  container: void 0,
  // Blur field when date is selected
  blurFieldOnSelect: true,
  // internationalization
  i18n: {
    previousMonth: "Previous Month",
    nextMonth: "Next Month",
    months: ["January", "February", "March", "April", "May", "June", "July", "August", "September", "October", "November", "December"],
    weekdays: ["Sunday", "Monday", "Tuesday", "Wednesday", "Thursday", "Friday", "Saturday"],
    weekdaysShort: ["Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat"]
  },
  // Theme Classname
  theme: null,
  // events array
  events: [],
  // callback function
  onSelect: null,
  onOpen: null,
  onClose: null,
  onDraw: null,
  // Enable keyboard input
  keyboardInput: true
};
const renderDayName = function(opts, day, abbr) {
  day += opts.firstDay;
  while (day >= 7) {
    day -= 7;
  }
  return abbr ? opts.i18n.weekdaysShort[day] : opts.i18n.weekdays[day];
};
const renderDay = function(opts) {
  let arr = [];
  let ariaSelected = "false";
  if (opts.isEmpty) {
    if (opts.showDaysInNextAndPreviousMonths) {
      arr.push("is-outside-current-month");
      if (!opts.enableSelectionDaysInNextAndPreviousMonths) {
        arr.push("is-selection-disabled");
      }
    } else {
      return '<td class="is-empty"></td>';
    }
  }
  if (opts.isDisabled) {
    arr.push("is-disabled");
  }
  if (opts.isToday) {
    arr.push("is-today");
  }
  if (opts.isSelected) {
    arr.push("is-selected");
    ariaSelected = "true";
  }
  if (opts.hasEvent) {
    arr.push("has-event");
  }
  if (opts.isInRange) {
    arr.push("is-inrange");
  }
  if (opts.isStartRange) {
    arr.push("is-startrange");
  }
  if (opts.isEndRange) {
    arr.push("is-endrange");
  }
  return '<td data-day="' + opts.day + '" class="' + arr.join(" ") + '" aria-selected="' + ariaSelected + '"><button class="pika-button pika-day" type="button" data-pika-year="' + opts.year + '" data-pika-month="' + opts.month + '" data-pika-day="' + opts.day + '">' + opts.day + "</button></td>";
};
const isoWeek = function(date, firstWeekOfYearMinDays) {
  date.setHours(0, 0, 0, 0);
  let yearDay = date.getDate(), weekDay = date.getDay(), dayInFirstWeek = firstWeekOfYearMinDays, dayShift = dayInFirstWeek - 1, daysPerWeek = 7, prevWeekDay = function(day) {
    return (day + daysPerWeek - 1) % daysPerWeek;
  };
  date.setDate(yearDay + dayShift - prevWeekDay(weekDay));
  let jan4th = new Date(date.getFullYear(), 0, dayInFirstWeek), msPerDay = 24 * 60 * 60 * 1e3, daysBetween = (date.getTime() - jan4th.getTime()) / msPerDay, weekNum = 1 + Math.round((daysBetween - dayShift + prevWeekDay(jan4th.getDay())) / daysPerWeek);
  return weekNum;
};
const renderWeek = function(pikaday, d, m, y, firstWeekOfYearMinDays) {
  let date = new Date(y, m, d), week = pikaday.hasMoment ? pikaday.moment(date).isoWeek() : isoWeek(date, firstWeekOfYearMinDays);
  return '<td class="pika-week">' + week + "</td>";
};
const renderRow = function(days, isRTL, pickWholeWeek, isRowSelected) {
  return '<tr class="pika-row' + (pickWholeWeek ? " pick-whole-week" : "") + (isRowSelected ? " is-selected" : "") + '">' + (isRTL ? days.reverse() : days).join("") + "</tr>";
};
const renderBody = function(rows) {
  return "<tbody>" + rows.join("") + "</tbody>";
};
const renderHead = function(opts) {
  let i, arr = [];
  if (opts.showWeekNumber) {
    arr.push("<th></th>");
  }
  for (i = 0; i < 7; i++) {
    arr.push('<th scope="col"><abbr title="' + renderDayName(opts, i) + '">' + renderDayName(opts, i, true) + "</abbr></th>");
  }
  return "<thead><tr>" + (opts.isRTL ? arr.reverse() : arr).join("") + "</tr></thead>";
};
const renderTitle = function(instance, c, year, month, refYear, randId) {
  let i, j, arr, opts = instance._o, isMinYear = year === opts.minYear, isMaxYear = year === opts.maxYear, html = '<div id="' + randId + '" class="pika-title" role="heading" aria-live="polite">', monthHtml, yearHtml, prev = true, next = true;
  for (arr = [], i = 0; i < 12; i++) {
    arr.push('<option value="' + (year === refYear ? i - c : 12 + i - c) + '"' + (i === month ? ' selected="selected"' : "") + (isMinYear && i < opts.minMonth || isMaxYear && i > opts.maxMonth ? ' disabled="disabled"' : "") + ">" + opts.i18n.months[i] + "</option>");
  }
  monthHtml = '<div class="pika-label">' + opts.i18n.months[month] + '<select class="pika-select pika-select-month" tabindex="-1">' + arr.join("") + "</select></div>";
  if (isArray(opts.yearRange)) {
    i = opts.yearRange[0];
    j = opts.yearRange[1] + 1;
  } else {
    i = year - opts.yearRange;
    j = 1 + year + opts.yearRange;
  }
  for (arr = []; i < j && i <= opts.maxYear; i++) {
    if (i >= opts.minYear) {
      arr.push('<option value="' + i + '"' + (i === year ? ' selected="selected"' : "") + ">" + i + "</option>");
    }
  }
  yearHtml = '<div class="pika-label">' + year + opts.yearSuffix + '<select class="pika-select pika-select-year" tabindex="-1">' + arr.join("") + "</select></div>";
  if (opts.showMonthAfterYear) {
    html += yearHtml + monthHtml;
  } else {
    html += monthHtml + yearHtml;
  }
  if (isMinYear && (month === 0 || opts.minMonth >= month)) {
    prev = false;
  }
  if (isMaxYear && (month === 11 || opts.maxMonth <= month)) {
    next = false;
  }
  if (c === 0) {
    html += '<button class="pika-prev' + (prev ? "" : " is-disabled") + '" type="button">' + opts.i18n.previousMonth + "</button>";
  }
  if (c === instance._o.numberOfMonths - 1) {
    html += '<button class="pika-next' + (next ? "" : " is-disabled") + '" type="button">' + opts.i18n.nextMonth + "</button>";
  }
  return html += "</div>";
};
const renderTable = function(opts, data, randId) {
  return '<table cellpadding="0" cellspacing="0" class="pika-table" role="grid" aria-labelledby="' + randId + '">' + renderHead(opts) + renderBody(data) + "</table>";
};
function Pikaday(options) {
  let self = this, opts = self.config(options);
  self._onMouseDown = function(e) {
    if (!self._v) {
      return;
    }
    e = e || getWindow().event;
    let target = e.target || e.srcElement;
    if (!target) {
      return;
    }
    if (!hasClass(target, "is-disabled")) {
      if (hasClass(target, "pika-button") && !hasClass(target, "is-empty") && !hasClass(target.parentNode, "is-disabled")) {
        self.setDate(
          new Date(
            target.getAttribute("data-pika-year"),
            target.getAttribute("data-pika-month"),
            target.getAttribute("data-pika-day")
          )
        );
        if (opts.bound) {
          sto(function() {
            self.hide();
            if (opts.blurFieldOnSelect && opts.field) {
              opts.field.blur();
            }
          }, 100);
        }
      } else if (hasClass(target, "pika-prev")) {
        self.prevMonth();
      } else if (hasClass(target, "pika-next")) {
        self.nextMonth();
      }
    }
    if (!hasClass(target, "pika-select")) {
      if (e.preventDefault) {
        e.preventDefault();
      } else {
        e.returnValue = false;
        return false;
      }
    } else {
      self._c = true;
    }
  };
  self._onChange = function(e) {
    e = e || getWindow().event;
    let target = e.target || e.srcElement;
    if (!target) {
      return;
    }
    if (hasClass(target, "pika-select-month")) {
      self.gotoMonth(target.value);
    } else if (hasClass(target, "pika-select-year")) {
      self.gotoYear(target.value);
    }
  };
  self._onKeyChange = function(e) {
    e = e || getWindow().event;
    if (self.isVisible()) {
      switch (e.keyCode) {
        case 13:
        case 27:
          if (opts.field) {
            opts.field.blur();
          }
          break;
        case 37:
          self.adjustDate("subtract", 1);
          break;
        case 38:
          self.adjustDate("subtract", 7);
          break;
        case 39:
          self.adjustDate("add", 1);
          break;
        case 40:
          self.adjustDate("add", 7);
          break;
        case 8:
        case 46:
          self.setDate(null);
          break;
      }
    }
  };
  self._parseFieldValue = function() {
    if (opts.parse) {
      return opts.parse(opts.field.value, opts.format);
    } else if (this.hasMoment) {
      let date = this.moment(
        opts.field.value,
        opts.format,
        opts.formatStrict
      );
      return date && date.isValid() ? date.toDate() : null;
    } else {
      return new Date(Date.parse(opts.field.value));
    }
  };
  self._onInputChange = function(e) {
    let date;
    if (e.firedBy === self) {
      return;
    }
    date = self._parseFieldValue();
    if (isDate(date)) {
      self.setDate(date);
    }
    if (!self._v) {
      self.show();
    }
  };
  self._onInputFocus = function() {
    self.show();
  };
  self._onInputClick = function() {
    self.show();
  };
  self._onInputBlur = function() {
    let pEl = getDocument().activeElement;
    do {
      if (hasClass(pEl, "pika-single")) {
        return;
      }
    } while (pEl = pEl.parentNode);
    if (!self._c) {
      self._b = sto(function() {
        self.hide();
      }, 50);
    }
    self._c = false;
  };
  self._onClick = function(e) {
    e = e || getWindow().event;
    let target = e.target || e.srcElement, pEl = target;
    if (!target) {
      return;
    }
    if (!hasEventListeners && hasClass(target, "pika-select")) {
      if (!target.onchange) {
        target.setAttribute("onchange", "return;");
        addEvent(target, "change", self._onChange);
      }
    }
    do {
      if (hasClass(pEl, "pika-single") || pEl === opts.trigger) {
        return;
      }
    } while (pEl = pEl.parentNode);
    if (self._v && target !== opts.trigger && pEl !== opts.trigger) {
      self.hide();
    }
  };
  self.el = getDocument().createElement("div");
  self.el.className = "pika-single" + (opts.isRTL ? " is-rtl" : "") + (opts.theme ? " " + opts.theme : "");
  addEvent(self.el, "mousedown", self._onMouseDown, true);
  addEvent(self.el, "touchend", self._onMouseDown, true);
  addEvent(self.el, "change", self._onChange);
  if (opts.keyboardInput) {
    addEvent(getDocument(), "keydown", self._onKeyChange);
  }
  if (opts.field) {
    if (opts.container) {
      opts.container.appendChild(self.el);
    } else if (opts.bound) {
      getDocument().body.appendChild(self.el);
    } else {
      opts.field.parentNode.insertBefore(
        self.el,
        opts.field.nextSibling
      );
    }
    addEvent(opts.field, "change", self._onInputChange);
    if (!opts.defaultDate) {
      opts.defaultDate = self._parseFieldValue();
      opts.setDefaultDate = true;
    }
  }
  let defDate = opts.defaultDate;
  if (isDate(defDate)) {
    if (opts.setDefaultDate) {
      self.setDate(defDate, true);
    } else {
      self.gotoDate(defDate);
    }
  } else {
    self.gotoDate(/* @__PURE__ */ new Date());
  }
  if (opts.bound) {
    this.hide();
    self.el.className += " is-bound";
    addEvent(opts.trigger, "click", self._onInputClick);
    addEvent(opts.trigger, "focus", self._onInputFocus);
    addEvent(opts.trigger, "blur", self._onInputBlur);
  } else {
    this.show();
  }
}
Pikaday.prototype = {
  /**
   * configure functionality
   */
  config: function(options) {
    if (!this._o) {
      this._o = extend({}, defaults, true);
    }
    let opts = extend(this._o, options, true);
    opts.isRTL = !!opts.isRTL;
    opts.field = opts.field && opts.field.nodeName ? opts.field : null;
    opts.theme = typeof opts.theme === "string" && opts.theme ? opts.theme : null;
    opts.bound = !!(opts.bound !== void 0 ? opts.field && opts.bound : opts.field);
    opts.trigger = opts.trigger && opts.trigger.nodeName ? opts.trigger : opts.field;
    opts.disableWeekends = !!opts.disableWeekends;
    opts.disableDayFn = typeof opts.disableDayFn === "function" ? opts.disableDayFn : null;
    let nom = parseInt(opts.numberOfMonths, 10) || 1;
    opts.numberOfMonths = nom > 4 ? 4 : nom;
    if (!isDate(opts.minDate)) {
      opts.minDate = false;
    }
    if (!isDate(opts.maxDate)) {
      opts.maxDate = false;
    }
    if (opts.minDate && opts.maxDate && opts.maxDate < opts.minDate) {
      opts.maxDate = opts.minDate = false;
    }
    if (opts.minDate) {
      this.setMinDate(opts.minDate);
    }
    if (opts.maxDate) {
      this.setMaxDate(opts.maxDate);
    }
    if (isArray(opts.yearRange)) {
      let fallback = (/* @__PURE__ */ new Date()).getFullYear() - 10;
      opts.yearRange[0] = parseInt(opts.yearRange[0], 10) || fallback;
      opts.yearRange[1] = parseInt(opts.yearRange[1], 10) || fallback;
    } else {
      opts.yearRange = Math.abs(parseInt(opts.yearRange, 10)) || defaults.yearRange;
      if (opts.yearRange > 100) {
        opts.yearRange = 100;
      }
    }
    return opts;
  },
  /**
   * return a formatted string of the current selection (using Moment.js if available)
   */
  toString: function(format) {
    format = format || this._o.format;
    if (!isDate(this._d)) {
      return "";
    }
    if (this._o.toString) {
      return this._o.toString(this._d, format);
    }
    if (this.hasMoment) {
      return this.moment(this._d).format(format);
    }
    return this._d.toDateString();
  },
  /**
   * return a Moment.js object of the current selection (if available)
   */
  getMoment: function() {
    return this.hasMoment ? this.moment(this._d) : null;
  },
  /**
   * set the current selection from a Moment.js object (if available)
   */
  setMoment: function(date, preventOnSelect) {
    if (this.hasMoment && this.moment.isMoment(date)) {
      this.setDate(date.toDate(), preventOnSelect);
    }
  },
  /**
   * Sets the momentjs function as dependency to parse and format the dates
   */
  useMoment: function(momentFunction) {
    this.hasMoment = true;
    this.moment = momentFunction;
  },
  /**
   * return a Date object of the current selection
   */
  getDate: function() {
    return isDate(this._d) ? new Date(this._d.getTime()) : null;
  },
  /**
   * set the current selection
   */
  setDate: function(date, preventOnSelect) {
    if (!date) {
      this._d = null;
      if (this._o.field) {
        this._o.field.value = "";
        fireEvent(this._o.field, "change", { firedBy: this });
      }
      return this.draw();
    }
    if (typeof date === "string") {
      date = new Date(Date.parse(date));
    }
    if (!isDate(date)) {
      return;
    }
    let min = this._o.minDate, max = this._o.maxDate;
    if (isDate(min) && date < min) {
      date = min;
    } else if (isDate(max) && date > max) {
      date = max;
    }
    this._d = new Date(date.getTime());
    setToStartOfDay(this._d);
    this.gotoDate(this._d);
    if (this._o.field) {
      this._o.field.value = this.toString();
      fireEvent(this._o.field, "change", { firedBy: this });
    }
    if (!preventOnSelect && typeof this._o.onSelect === "function") {
      this._o.onSelect.call(this, this.getDate());
    }
  },
  /**
   * clear and reset the date
   */
  clear: function() {
    this.setDate(null);
  },
  /**
   * change view to a specific date
   */
  gotoDate: function(date) {
    let newCalendar = true;
    if (!isDate(date)) {
      return;
    }
    if (this.calendars) {
      let firstVisibleDate = new Date(this.calendars[0].year, this.calendars[0].month, 1), lastVisibleDate = new Date(this.calendars[this.calendars.length - 1].year, this.calendars[this.calendars.length - 1].month, 1), visibleDate = date.getTime();
      lastVisibleDate.setMonth(lastVisibleDate.getMonth() + 1);
      lastVisibleDate.setDate(lastVisibleDate.getDate() - 1);
      newCalendar = visibleDate < firstVisibleDate.getTime() || lastVisibleDate.getTime() < visibleDate;
    }
    if (newCalendar) {
      this.calendars = [{
        month: date.getMonth(),
        year: date.getFullYear()
      }];
      if (this._o.mainCalendar === "right") {
        this.calendars[0].month += 1 - this._o.numberOfMonths;
      }
    }
    this.adjustCalendars();
  },
  adjustDate: function(sign, days) {
    let day = this.getDate() || /* @__PURE__ */ new Date();
    let difference = parseInt(days) * 24 * 60 * 60 * 1e3;
    let newDay;
    if (sign === "add") {
      newDay = new Date(day.valueOf() + difference);
    } else if (sign === "subtract") {
      newDay = new Date(day.valueOf() - difference);
    }
    this.setDate(newDay);
  },
  adjustCalendars: function() {
    this.calendars[0] = adjustCalendar(this.calendars[0]);
    for (let c = 1; c < this._o.numberOfMonths; c++) {
      this.calendars[c] = adjustCalendar({
        month: this.calendars[0].month + c,
        year: this.calendars[0].year
      });
    }
    this.draw();
  },
  gotoToday: function() {
    this.gotoDate(/* @__PURE__ */ new Date());
  },
  /**
   * change view to a specific month (zero-index, e.g. 0: January)
   */
  gotoMonth: function(month) {
    if (!isNaN(month)) {
      this.calendars[0].month = parseInt(month, 10);
      this.adjustCalendars();
    }
  },
  nextMonth: function() {
    this.calendars[0].month++;
    this.adjustCalendars();
  },
  prevMonth: function() {
    this.calendars[0].month--;
    this.adjustCalendars();
  },
  /**
   * change view to a specific full year (e.g. "2012")
   */
  gotoYear: function(year) {
    if (!isNaN(year)) {
      this.calendars[0].year = parseInt(year, 10);
      this.adjustCalendars();
    }
  },
  /**
   * change the minDate
   */
  setMinDate: function(value) {
    if (value instanceof Date) {
      setToStartOfDay(value);
      this._o.minDate = value;
      this._o.minYear = value.getFullYear();
      this._o.minMonth = value.getMonth();
    } else {
      this._o.minDate = defaults.minDate;
      this._o.minYear = defaults.minYear;
      this._o.minMonth = defaults.minMonth;
      this._o.startRange = defaults.startRange;
    }
    this.draw();
  },
  /**
   * change the maxDate
   */
  setMaxDate: function(value) {
    if (value instanceof Date) {
      setToStartOfDay(value);
      this._o.maxDate = value;
      this._o.maxYear = value.getFullYear();
      this._o.maxMonth = value.getMonth();
    } else {
      this._o.maxDate = defaults.maxDate;
      this._o.maxYear = defaults.maxYear;
      this._o.maxMonth = defaults.maxMonth;
      this._o.endRange = defaults.endRange;
    }
    this.draw();
  },
  setStartRange: function(value) {
    this._o.startRange = value;
  },
  setEndRange: function(value) {
    this._o.endRange = value;
  },
  /**
   * refresh the HTML
   */
  draw: function(force) {
    if (!this._v && !force) {
      return;
    }
    let opts = this._o, minYear = opts.minYear, maxYear = opts.maxYear, minMonth = opts.minMonth, maxMonth = opts.maxMonth, html = "", randId;
    if (this._y <= minYear) {
      this._y = minYear;
      if (!isNaN(minMonth) && this._m < minMonth) {
        this._m = minMonth;
      }
    }
    if (this._y >= maxYear) {
      this._y = maxYear;
      if (!isNaN(maxMonth) && this._m > maxMonth) {
        this._m = maxMonth;
      }
    }
    for (let c = 0; c < opts.numberOfMonths; c++) {
      randId = "pika-title-" + Math.random().toString(36).replace(/[^a-z]+/g, "").substr(0, 2);
      html += '<div class="pika-lendar">' + renderTitle(this, c, this.calendars[c].year, this.calendars[c].month, this.calendars[0].year, randId) + this.render(this.calendars[c].year, this.calendars[c].month, randId) + "</div>";
    }
    this.el.innerHTML = html;
    if (opts.bound) {
      if (opts.field.type !== "hidden") {
        sto(function() {
          opts.trigger.focus();
        }, 1);
      }
    }
    if (typeof this._o.onDraw === "function") {
      this._o.onDraw(this);
    }
    if (opts.bound) {
      opts.field.setAttribute("aria-label", opts.ariaLabel);
    }
  },
  adjustPosition: function() {
    let field, width, height, viewportWidth, viewportHeight, scrollTop, left, top, clientRect, leftAligned, bottomAligned;
    if (this._o.container) {
      return;
    }
    this.el.style.position = "absolute";
    field = this._o.trigger;
    width = this.el.offsetWidth;
    height = this.el.offsetHeight;
    viewportWidth = getWindow().innerWidth || getDocument().documentElement.clientWidth;
    viewportHeight = getWindow().innerHeight || getDocument().documentElement.clientHeight;
    scrollTop = getDocument().body.scrollTop || getDocument().documentElement.scrollTop;
    leftAligned = true;
    bottomAligned = true;
    clientRect = field.getBoundingClientRect();
    left = clientRect.left + getWindow().pageXOffset;
    top = clientRect.bottom + getWindow().pageYOffset;
    if (this._o.reposition && left + width > viewportWidth || this._o.position.indexOf("right") > -1 && left - width + field.offsetWidth > 0) {
      left = left - width + field.offsetWidth;
      leftAligned = false;
    }
    if (this._o.reposition && top + height > viewportHeight + scrollTop || this._o.position.indexOf("top") > -1 && top - height - field.offsetHeight > 0) {
      top = top - height - field.offsetHeight;
      bottomAligned = false;
    }
    if (left < 0) {
      left = 0;
    }
    if (top < 0) {
      top = 0;
    }
    this.el.style.left = left + "px";
    this.el.style.top = top + "px";
    addClass(this.el, leftAligned ? "left-aligned" : "right-aligned");
    addClass(this.el, bottomAligned ? "bottom-aligned" : "top-aligned");
    removeClass(this.el, !leftAligned ? "left-aligned" : "right-aligned");
    removeClass(this.el, !bottomAligned ? "bottom-aligned" : "top-aligned");
  },
  /**
   * render HTML for a particular month
   */
  render: function(year, month, randId) {
    let opts = this._o, now = /* @__PURE__ */ new Date(), days = getDaysInMonth(year, month), before = new Date(year, month, 1).getDay(), data = [], row = [];
    setToStartOfDay(now);
    if (opts.firstDay > 0) {
      before -= opts.firstDay;
      if (before < 0) {
        before += 7;
      }
    }
    let previousMonth = month === 0 ? 11 : month - 1, nextMonth = month === 11 ? 0 : month + 1, yearOfPreviousMonth = month === 0 ? year - 1 : year, yearOfNextMonth = month === 11 ? year + 1 : year, daysInPreviousMonth = getDaysInMonth(yearOfPreviousMonth, previousMonth);
    let cells = days + before, after = cells;
    while (after > 7) {
      after -= 7;
    }
    cells += 7 - after;
    let isWeekSelected = false;
    for (let i = 0, r = 0; i < cells; i++) {
      let day = new Date(year, month, 1 + (i - before)), isSelected = isDate(this._d) ? compareDates(day, this._d) : false, isToday = compareDates(day, now), hasEvent = opts.events.indexOf(day.toDateString()) !== -1 ? true : false, isEmpty = i < before || i >= days + before, dayNumber = 1 + (i - before), monthNumber = month, yearNumber = year, isStartRange = opts.startRange && compareDates(opts.startRange, day), isEndRange = opts.endRange && compareDates(opts.endRange, day), isInRange = opts.startRange && opts.endRange && opts.startRange < day && day < opts.endRange, isDisabled = opts.minDate && day < opts.minDate || opts.maxDate && day > opts.maxDate || opts.disableWeekends && isWeekend(day) || opts.disableDayFn && opts.disableDayFn(day);
      if (isEmpty) {
        if (i < before) {
          dayNumber = daysInPreviousMonth + dayNumber;
          monthNumber = previousMonth;
          yearNumber = yearOfPreviousMonth;
        } else {
          dayNumber = dayNumber - days;
          monthNumber = nextMonth;
          yearNumber = yearOfNextMonth;
        }
      }
      let dayConfig = {
        day: dayNumber,
        month: monthNumber,
        year: yearNumber,
        hasEvent,
        isSelected,
        isToday,
        isDisabled,
        isEmpty,
        isStartRange,
        isEndRange,
        isInRange,
        showDaysInNextAndPreviousMonths: opts.showDaysInNextAndPreviousMonths,
        enableSelectionDaysInNextAndPreviousMonths: opts.enableSelectionDaysInNextAndPreviousMonths
      };
      if (opts.pickWholeWeek && isSelected) {
        isWeekSelected = true;
      }
      row.push(renderDay(dayConfig));
      if (++r === 7) {
        if (opts.showWeekNumber) {
          row.unshift(renderWeek(this, i - before, month, year, opts.firstWeekOfYearMinDays));
        }
        data.push(renderRow(row, opts.isRTL, opts.pickWholeWeek, isWeekSelected));
        row = [];
        r = 0;
        isWeekSelected = false;
      }
    }
    return renderTable(opts, data, randId);
  },
  isVisible: function() {
    return this._v;
  },
  show: function() {
    if (!this.isVisible()) {
      this._v = true;
      this.draw();
      removeClass(this.el, "is-hidden");
      if (this._o.bound) {
        addEvent(getDocument(), "click", this._onClick);
        this.adjustPosition();
      }
      if (typeof this._o.onOpen === "function") {
        this._o.onOpen.call(this);
      }
    }
  },
  hide: function() {
    let v = this._v;
    if (v !== false) {
      if (this._o.bound) {
        removeEvent(getDocument(), "click", this._onClick);
      }
      if (!this._o.container) {
        this.el.style.position = "static";
        this.el.style.left = "auto";
        this.el.style.top = "auto";
      }
      addClass(this.el, "is-hidden");
      this._v = false;
      if (v !== void 0 && typeof this._o.onClose === "function") {
        this._o.onClose.call(this);
      }
    }
  },
  /**
   * GAME OVER
   */
  destroy: function() {
    let opts = this._o;
    this.hide();
    removeEvent(this.el, "mousedown", this._onMouseDown, true);
    removeEvent(this.el, "touchend", this._onMouseDown, true);
    removeEvent(this.el, "change", this._onChange);
    if (opts.keyboardInput) {
      removeEvent(getDocument(), "keydown", this._onKeyChange);
    }
    if (opts.field) {
      removeEvent(opts.field, "change", this._onInputChange);
      if (opts.bound) {
        removeEvent(opts.trigger, "click", this._onInputClick);
        removeEvent(opts.trigger, "focus", this._onInputFocus);
        removeEvent(opts.trigger, "blur", this._onInputBlur);
      }
    }
    if (this.el.parentNode) {
      this.el.parentNode.removeChild(this.el);
    }
  }
};
export {
  Pikaday as default
};
