"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * Handles async tasks.
 */
class AsyncTaskManager {
    constructor() {
        this.runningTasks = [];
        this.runningTimers = [];
        this.queue = [];
    }
    /**
     * Returns a promise that is fulfilled when async tasks are complete.
     * This method is not part of the HTML standard.
     *
     * @returns Promise.
     */
    async whenComplete() {
        return new Promise((resolve, reject) => {
            const timerID = global.setTimeout(() => {
                this.endTimer(timerID);
            }, 0);
            this.startTimer(timerID);
            this.queue.push({ resolve, reject });
        });
    }
    /**
     * Cancels all tasks.
     *
     * @param [error] Error.
     */
    cancelAll(error) {
        for (const timerID of this.runningTimers) {
            global.clearTimeout(timerID);
        }
        const promises = this.queue;
        this.runningTasks = [];
        this.runningTimers = [];
        this.queue = [];
        for (const promise of promises) {
            if (error) {
                promise.reject(error);
            }
            else {
                promise.resolve();
            }
        }
    }
    /**
     * Starts a timer.
     *
     * @param timerID Timer ID.
     */
    startTimer(timerID) {
        this.runningTimers.push(timerID);
    }
    /**
     * Ends a timer.
     *
     * @param timerID Timer ID.
     */
    endTimer(timerID) {
        const index = this.runningTimers.indexOf(timerID);
        if (index !== -1) {
            this.runningTimers.splice(index, 1);
        }
        if (!this.runningTasks.length && !this.runningTimers.length) {
            this.cancelAll();
        }
    }
    /**
     * Starts an async task.
     *
     * @returns Task ID.
     */
    startTask() {
        const taskID = this.newTaskID();
        this.runningTasks.push(taskID);
        return taskID;
    }
    /**
     * Ends an async task.
     *
     * @param taskID Task ID.
     */
    endTask(taskID) {
        const index = this.runningTasks.indexOf(taskID);
        if (index !== -1) {
            this.runningTasks.splice(index, 1);
        }
        if (!this.runningTasks.length && !this.runningTimers.length) {
            this.cancelAll();
        }
    }
    /**
     * Returns the amount of running tasks.
     *
     * @returns Count.
     */
    getTaskCount() {
        return this.runningTasks.length;
    }
    /**
     * Returns the amount of running timers.
     *
     * @returns Count.
     */
    getTimerCount() {
        return this.runningTimers.length;
    }
    /**
     * Returns a new task ID.
     *
     * @returns Task ID.
     */
    newTaskID() {
        this.constructor.taskID++;
        return this.constructor.taskID;
    }
}
exports.default = AsyncTaskManager;
AsyncTaskManager.taskID = 0;
//# sourceMappingURL=AsyncTaskManager.js.map