"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const CSSRule_1 = __importDefault(require("./CSSRule"));
const CSSStyleRule_1 = __importDefault(require("./rules/CSSStyleRule"));
const CSSKeyframeRule_1 = __importDefault(require("./rules/CSSKeyframeRule"));
const CSSKeyframesRule_1 = __importDefault(require("./rules/CSSKeyframesRule"));
const CSSMediaRule_1 = __importDefault(require("./rules/CSSMediaRule"));
const CSSStyleDeclaration_1 = __importDefault(require("./CSSStyleDeclaration"));
const COMMENT_REGEXP = /\/\*[^*]*\*\//gm;
/**
 * CSS parser.
 */
class CSSParser {
    /**
     * Parses HTML and returns a root element.
     *
     * @param parentStyleSheet Parent style sheet.
     * @param cssText CSS code.
     * @returns Root element.
     */
    static parseFromString(parentStyleSheet, cssText) {
        const css = cssText.replace(COMMENT_REGEXP, '');
        const cssRules = [];
        const regExp = /{|}/gm;
        const stack = [];
        let parentRule = null;
        let lastIndex = 0;
        let match;
        while ((match = regExp.exec(css))) {
            if (match[0] === '{') {
                const selectorText = css.substring(lastIndex, match.index).trim();
                if (selectorText.startsWith('@keyframes')) {
                    const newRule = new CSSKeyframesRule_1.default();
                    newRule.name = selectorText.replace('@keyframes ', '');
                    newRule.parentStyleSheet = parentStyleSheet;
                    cssRules.push(newRule);
                    parentRule = newRule;
                }
                else if (selectorText.startsWith('@media')) {
                    const mediums = selectorText.replace('@media', '').split(',');
                    const newRule = new CSSMediaRule_1.default();
                    for (const medium of mediums) {
                        newRule.media.appendMedium(medium.trim());
                    }
                    newRule.parentStyleSheet = parentStyleSheet;
                    cssRules.push(newRule);
                    parentRule = newRule;
                }
                else if (parentRule && parentRule.type === CSSRule_1.default.KEYFRAMES_RULE) {
                    const newRule = new CSSKeyframeRule_1.default();
                    newRule.keyText = selectorText.trim();
                    newRule.parentStyleSheet = parentStyleSheet;
                    newRule.parentRule = parentRule;
                    parentRule.cssRules.push(newRule);
                    parentRule = newRule;
                }
                else if (parentRule && parentRule.type === CSSRule_1.default.MEDIA_RULE) {
                    const newRule = new CSSStyleRule_1.default();
                    newRule.selectorText = selectorText;
                    newRule.parentStyleSheet = parentStyleSheet;
                    newRule.parentRule = parentRule;
                    parentRule.cssRules.push(newRule);
                    parentRule = newRule;
                }
                else {
                    const newRule = new CSSStyleRule_1.default();
                    newRule.selectorText = selectorText;
                    newRule.parentStyleSheet = parentStyleSheet;
                    newRule.parentRule = parentRule;
                    if (!parentRule) {
                        cssRules.push(newRule);
                    }
                    parentRule = newRule;
                }
                stack.push(parentRule);
            }
            else {
                if (parentRule) {
                    const cssText = css.substring(lastIndex, match.index).trim();
                    switch (parentRule.type) {
                        case CSSRule_1.default.FONT_FACE_RULE:
                        case CSSRule_1.default.KEYFRAME_RULE:
                        case CSSRule_1.default.STYLE_RULE:
                            const style = new CSSStyleDeclaration_1.default();
                            style.cssText = cssText;
                            style.parentRule = parentRule;
                            parentRule.style = style;
                            break;
                    }
                }
                stack.pop();
                parentRule = stack[stack.length - 1] || null;
            }
            lastIndex = match.index + 1;
        }
        return cssRules;
    }
}
exports.default = CSSParser;
//# sourceMappingURL=CSSParser.js.map