import Node from '../node/Node';
import Attr from '../../attribute/Attr';
import DOMRect from './DOMRect';
import IDOMTokenList from '../../dom-token-list/IDOMTokenList';
import IElement from './IElement';
import IShadowRoot from '../shadow-root/IShadowRoot';
import INode from '../node/INode';
import IHTMLCollection from './IHTMLCollection';
import INodeList from '../node/INodeList';
import { TInsertAdjacentPositions } from './IElement';
import IDOMRectList from './IDOMRectList';
/**
 * Element.
 */
export default class Element extends Node implements IElement {
    static _observedAttributes: string[];
    static observedAttributes: string[];
    tagName: string;
    nodeType: import("../node/NodeTypeEnum").default;
    shadowRoot: IShadowRoot;
    scrollTop: number;
    scrollLeft: number;
    children: IHTMLCollection<IElement>;
    readonly namespaceURI: string;
    _shadowRoot: IShadowRoot;
    _attributes: {
        [k: string]: Attr;
    };
    private _classList;
    _isValue?: string;
    /**
     * Returns class list.
     *
     * @returns Class list.
     */
    get classList(): IDOMTokenList;
    /**
     * Returns ID.
     *
     * @returns ID.
     */
    get id(): string;
    /**
     * Sets ID.
     *
     * @param id ID.
     */
    set id(id: string);
    /**
     * Returns class name.
     *
     * @returns Class name.
     */
    get className(): string;
    /**
     * Sets class name.
     *
     * @param className Class name.
     */
    set className(className: string);
    /**
     * Node name.
     *
     * @returns Node name.
     */
    get nodeName(): string;
    /**
     * Local name.
     *
     * @returns Local name.
     */
    get localName(): string;
    /**
     * Previous element sibling.
     *
     * @returns Element.
     */
    get previousElementSibling(): IElement;
    /**
     * Next element sibling.
     *
     * @returns Element.
     */
    get nextElementSibling(): IElement;
    /**
     * Get text value of children.
     *
     * @returns Text content.
     */
    get textContent(): string;
    /**
     * Sets text content.
     *
     * @param textContent Text content.
     */
    set textContent(textContent: string);
    /**
     * Returns inner HTML.
     *
     * @returns HTML.
     */
    get innerHTML(): string;
    /**
     * Sets inner HTML.
     *
     * @param html HTML.
     */
    set innerHTML(html: string);
    /**
     * Returns outer HTML.
     *
     * @returns HTML.
     */
    get outerHTML(): string;
    /**
     * Returns outer HTML.
     *
     * @param html HTML.
     */
    set outerHTML(html: string);
    /**
     * Returns attributes.
     *
     * @returns Attributes.
     */
    get attributes(): {
        [k: string]: Attr | number;
    };
    /**
     * First element child.
     *
     * @returns Element.
     */
    get firstElementChild(): IElement;
    /**
     * Last element child.
     *
     * @returns Element.
     */
    get lastElementChild(): IElement;
    /**
     * Last element child.
     *
     * @returns Element.
     */
    get childElementCount(): number;
    /**
     * Returns slot.
     *
     * @returns Slot.
     */
    get slot(): string;
    /**
     * Returns slot.
     *
     * @param slot Slot.
     */
    set slot(title: string);
    /**
     * Attribute changed callback.
     *
     * @param name Name.
     * @param oldValue Old value.
     * @param newValue New value.
     */
    attributeChangedCallback?(name: string, oldValue: string, newValue: string): void;
    /**
     * Returns inner HTML and optionally the content of shadow roots.
     *
     * This is a feature implemented in Chromium, but not supported by Mozilla yet.
     *
     * @see https://web.dev/declarative-shadow-dom/
     * @see https://chromestatus.com/feature/5191745052606464
     * @param [options] Options.
     * @param [options.includeShadowRoots] Set to "true" to include shadow roots.
     * @returns HTML.
     */
    getInnerHTML(options?: {
        includeShadowRoots?: boolean;
    }): string;
    /**
     * Clones a node.
     *
     * @override
     * @param [deep=false] "true" to clone deep.
     * @returns Cloned node.
     */
    cloneNode(deep?: boolean): IElement;
    /**
     * Append a child node to childNodes.
     *
     * @override
     * @param  node Node to append.
     * @returns Appended node.
     */
    appendChild(node: INode): INode;
    /**
     * Remove Child element from childNodes array.
     *
     * @override
     * @param node Node to remove.
     */
    removeChild(node: INode): INode;
    /**
     * Removes the node from its parent.
     */
    remove(): void;
    /**
     * Inserts a node before another.
     *
     * @override
     * @param newNode Node to insert.
     * @param [referenceNode] Node to insert before.
     * @returns Inserted node.
     */
    insertBefore(newNode: INode, referenceNode?: INode): INode;
    /**
     * The Node.replaceWith() method replaces this Node in the children list of its parent with a set of Node or DOMString objects.
     *
     * @param nodes List of Node or DOMString.
     */
    replaceWith(...nodes: (INode | string)[]): void;
    /**
     * Inserts a set of Node or DOMString objects in the children list of this ChildNode's parent, just before this ChildNode. DOMString objects are inserted as equivalent Text nodes.
     *
     * @param nodes List of Node or DOMString.
     */
    before(...nodes: (string | INode)[]): void;
    /**
     * Inserts a set of Node or DOMString objects in the children list of this ChildNode's parent, just after this ChildNode. DOMString objects are inserted as equivalent Text nodes.
     *
     * @param nodes List of Node or DOMString.
     */
    after(...nodes: (string | INode)[]): void;
    /**
     * Inserts a set of Node objects or DOMString objects after the last child of the ParentNode. DOMString objects are inserted as equivalent Text nodes.
     *
     * @param nodes List of Node or DOMString.
     */
    append(...nodes: (string | INode)[]): void;
    /**
     * Inserts a set of Node objects or DOMString objects before the first child of the ParentNode. DOMString objects are inserted as equivalent Text nodes.
     *
     * @param nodes List of Node or DOMString.
     */
    prepend(...nodes: (string | INode)[]): void;
    /**
     * Replaces the existing children of a node with a specified new set of children.
     *
     * @param nodes List of Node or DOMString.
     */
    replaceChildren(...nodes: (string | INode)[]): void;
    /**
     * Inserts a node to the given position.
     *
     * @param position Position to insert element.
     * @param element Node to insert.
     * @returns Inserted node or null if couldn't insert.
     */
    insertAdjacentElement(position: TInsertAdjacentPositions, element: INode): INode | null;
    /**
     * Inserts an HTML string to the given position.
     *
     * @param position Position to insert text.
     * @param text HTML string to insert.
     */
    insertAdjacentHTML(position: TInsertAdjacentPositions, text: string): void;
    /**
     * Inserts text to the given position.
     *
     * @param position Position to insert text.
     * @param text String to insert.
     */
    insertAdjacentText(position: TInsertAdjacentPositions, text: string): void;
    /**
     * Sets an attribute.
     *
     * @param name Name.
     * @param value Value.
     */
    setAttribute(name: string, value: string): void;
    /**
     * Sets a namespace attribute.
     *
     * @param namespaceURI Namespace URI.
     * @param name Name.
     * @param value Value.
     */
    setAttributeNS(namespaceURI: string, name: string, value: string): void;
    /**
     * Returns attribute names.
     *
     * @returns Attribute names.
     */
    getAttributeNames(): string[];
    /**
     * Returns attribute value.
     *
     * @param name Name.
     */
    getAttribute(name: string): string;
    /**
     * Returns namespace attribute value.
     *
     * @param namespace Namespace URI.
     * @param localName Local name.
     */
    getAttributeNS(namespace: string, localName: string): string;
    /**
     * Returns a boolean value indicating whether the specified element has the attribute or not.
     *
     * @param name Attribute name.
     * @returns True if attribute exists, false otherwise.
     */
    hasAttribute(name: string): boolean;
    /**
     * Returns a boolean value indicating whether the specified element has the namespace attribute or not.
     *
     * @param namespace Namespace URI.
     * @param localName Local name.
     * @returns True if attribute exists, false otherwise.
     */
    hasAttributeNS(namespace: string, localName: string): boolean;
    /**
     * Returns "true" if the element has attributes.
     *
     * @returns "true" if the element has attributes.
     */
    hasAttributes(): boolean;
    /**
     * Removes an attribute.
     *
     * @param name Name.
     */
    removeAttribute(name: string): void;
    /**
     * Removes a namespace attribute.
     *
     * @param namespace Namespace URI.
     * @param localName Local name.
     */
    removeAttributeNS(namespace: string, localName: string): void;
    /**
     * Attaches a shadow root.
     *
     * @param _shadowRootInit Shadow root init.
     * @param shadowRootInit
     * @param shadowRootInit.mode
     * @returns Shadow root.
     */
    attachShadow(shadowRootInit: {
        mode: string;
    }): IShadowRoot;
    /**
     * Converts to string.
     *
     * @returns String.
     */
    toString(): string;
    /**
     * Returns the size of an element and its position relative to the viewport.
     *
     * @returns DOM rect.
     */
    getBoundingClientRect(): DOMRect;
    /**
     * Returns a collection of DOMRect objects that indicate the bounding rectangles for each CSS border box in a client.
     *
     * @see https://developer.mozilla.org/en-US/docs/Web/API/Element/getClientRects
     * @returns DOM rect list.
     */
    getClientRects(): IDOMRectList<DOMRect>;
    /**
     * The matches() method checks to see if the Element would be selected by the provided selectorString.
     *
     * @param selector Selector.
     * @returns "true" if matching.
     */
    matches(selector: string): boolean;
    /**
     * Traverses the Element and its parents (heading toward the document root) until it finds a node that matches the provided selector string.
     *
     * @param selector Selector.
     * @returns Closest matching element.
     */
    closest(selector: string): IElement;
    /**
     * Query CSS selector to find matching nodes.
     *
     * @param selector CSS selector.
     * @returns Matching elements.
     */
    querySelectorAll(selector: string): INodeList<IElement>;
    /**
     * Query CSS Selector to find matching node.
     *
     * @param selector CSS selector.
     * @returns Matching element.
     */
    querySelector(selector: string): IElement;
    /**
     * Returns an elements by class name.
     *
     * @param className Tag name.
     * @returns Matching element.
     */
    getElementsByClassName(className: string): IHTMLCollection<IElement>;
    /**
     * Returns an elements by tag name.
     *
     * @param tagName Tag name.
     * @returns Matching element.
     */
    getElementsByTagName(tagName: string): IHTMLCollection<IElement>;
    /**
     * Returns an elements by tag name and namespace.
     *
     * @param namespaceURI Namespace URI.
     * @param tagName Tag name.
     * @returns Matching element.
     */
    getElementsByTagNameNS(namespaceURI: string, tagName: string): IHTMLCollection<IElement>;
    /**
     * The setAttributeNode() method adds a new Attr node to the specified element.
     *
     * @param attribute Attribute.
     * @returns Replaced attribute.
     */
    setAttributeNode(attribute: Attr): Attr;
    /**
     * The setAttributeNodeNS() method adds a new Attr node to the specified element.
     *
     * @param attribute Attribute.
     * @returns Replaced attribute.
     */
    setAttributeNodeNS(attribute: Attr): Attr;
    /**
     * Returns an Attr node.
     *
     * @param name Name.
     * @returns Replaced attribute.
     */
    getAttributeNode(name: string): Attr;
    /**
     * Returns a namespaced Attr node.
     *
     * @param namespace Namespace.
     * @param name Name.
     * @returns Replaced attribute.
     */
    getAttributeNodeNS(namespace: string, name: string): Attr;
    /**
     * Removes an Attr node.
     *
     * @param attribute Attribute.
     */
    removeAttributeNode(attribute: Attr): void;
    /**
     * Removes an Attr node.
     *
     * @param attribute Attribute.
     */
    removeAttributeNodeNS(attribute: Attr): void;
    /**
     * Scrolls to a particular set of coordinates.
     *
     * @param x X position or options object.
     * @param y Y position.
     */
    scroll(x: {
        top?: number;
        left?: number;
        behavior?: string;
    } | number, y?: number): void;
    /**
     * Scrolls to a particular set of coordinates.
     *
     * @param x X position or options object.
     * @param y Y position.
     */
    scrollTo(x: {
        top?: number;
        left?: number;
        behavior?: string;
    } | number, y?: number): void;
    /**
     * Returns attribute name.
     *
     * @param name Name.
     * @returns Attribute name based on namespace.
     */
    protected _getAttributeName(name: any): string;
    /**
     * Updates DOM list indices.
     */
    protected _updateDomListIndices(): void;
}
