"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const Node_1 = __importDefault(require("../node/Node"));
const ShadowRoot_1 = __importDefault(require("../shadow-root/ShadowRoot"));
const Attr_1 = __importDefault(require("../../attribute/Attr"));
const DOMRect_1 = __importDefault(require("./DOMRect"));
const DOMTokenList_1 = __importDefault(require("../../dom-token-list/DOMTokenList"));
const QuerySelector_1 = __importDefault(require("../../query-selector/QuerySelector"));
const SelectorItem_1 = __importDefault(require("../../query-selector/SelectorItem"));
const MutationRecord_1 = __importDefault(require("../../mutation-observer/MutationRecord"));
const MutationTypeEnum_1 = __importDefault(require("../../mutation-observer/MutationTypeEnum"));
const NamespaceURI_1 = __importDefault(require("../../config/NamespaceURI"));
const XMLParser_1 = __importDefault(require("../../xml-parser/XMLParser"));
const XMLSerializer_1 = __importDefault(require("../../xml-serializer/XMLSerializer"));
const ChildNodeUtility_1 = __importDefault(require("../child-node/ChildNodeUtility"));
const ParentNodeUtility_1 = __importDefault(require("../parent-node/ParentNodeUtility"));
const NonDocumentChildNodeUtility_1 = __importDefault(require("../child-node/NonDocumentChildNodeUtility"));
const DOMException_1 = __importDefault(require("../../exception/DOMException"));
const HTMLCollectionFactory_1 = __importDefault(require("./HTMLCollectionFactory"));
const DOMRectListFactory_1 = __importDefault(require("./DOMRectListFactory"));
/**
 * Element.
 */
class Element extends Node_1.default {
    constructor() {
        super(...arguments);
        this.tagName = null;
        this.nodeType = Node_1.default.ELEMENT_NODE;
        this.shadowRoot = null;
        this.scrollTop = 0;
        this.scrollLeft = 0;
        this.children = HTMLCollectionFactory_1.default.create();
        this.namespaceURI = null;
        // Used for being able to access closed shadow roots
        this._shadowRoot = null;
        this._attributes = {};
        this._classList = null;
    }
    /**
     * Returns class list.
     *
     * @returns Class list.
     */
    get classList() {
        if (!this._classList) {
            this._classList = new DOMTokenList_1.default(this, 'class');
        }
        return this._classList;
    }
    /**
     * Returns ID.
     *
     * @returns ID.
     */
    get id() {
        return this.getAttribute('id') || '';
    }
    /**
     * Sets ID.
     *
     * @param id ID.
     */
    set id(id) {
        this.setAttribute('id', id);
    }
    /**
     * Returns class name.
     *
     * @returns Class name.
     */
    get className() {
        return this.getAttribute('class') || '';
    }
    /**
     * Sets class name.
     *
     * @param className Class name.
     */
    set className(className) {
        this.setAttribute('class', className);
    }
    /**
     * Node name.
     *
     * @returns Node name.
     */
    get nodeName() {
        return this.tagName;
    }
    /**
     * Local name.
     *
     * @returns Local name.
     */
    get localName() {
        return this.tagName ? this.tagName.toLowerCase() : 'unknown';
    }
    /**
     * Previous element sibling.
     *
     * @returns Element.
     */
    get previousElementSibling() {
        return NonDocumentChildNodeUtility_1.default.previousElementSibling(this);
    }
    /**
     * Next element sibling.
     *
     * @returns Element.
     */
    get nextElementSibling() {
        return NonDocumentChildNodeUtility_1.default.nextElementSibling(this);
    }
    /**
     * Get text value of children.
     *
     * @returns Text content.
     */
    get textContent() {
        let result = '';
        for (const childNode of this.childNodes) {
            if (childNode.nodeType === Node_1.default.ELEMENT_NODE || childNode.nodeType === Node_1.default.TEXT_NODE) {
                result += childNode.textContent;
            }
        }
        return result;
    }
    /**
     * Sets text content.
     *
     * @param textContent Text content.
     */
    set textContent(textContent) {
        for (const child of this.childNodes.slice()) {
            this.removeChild(child);
        }
        if (textContent) {
            this.appendChild(this.ownerDocument.createTextNode(textContent));
        }
    }
    /**
     * Returns inner HTML.
     *
     * @returns HTML.
     */
    get innerHTML() {
        return this.getInnerHTML();
    }
    /**
     * Sets inner HTML.
     *
     * @param html HTML.
     */
    set innerHTML(html) {
        for (const child of this.childNodes.slice()) {
            this.removeChild(child);
        }
        for (const node of XMLParser_1.default.parse(this.ownerDocument, html).childNodes.slice()) {
            this.appendChild(node);
        }
    }
    /**
     * Returns outer HTML.
     *
     * @returns HTML.
     */
    get outerHTML() {
        return new XMLSerializer_1.default().serializeToString(this);
    }
    /**
     * Returns outer HTML.
     *
     * @param html HTML.
     */
    set outerHTML(html) {
        this.replaceWith(html);
    }
    /**
     * Returns attributes.
     *
     * @returns Attributes.
     */
    get attributes() {
        const attributes = Object.values(this._attributes);
        return Object.assign({}, this._attributes, attributes, {
            length: attributes.length
        });
    }
    /**
     * First element child.
     *
     * @returns Element.
     */
    get firstElementChild() {
        return this.children ? this.children[0] || null : null;
    }
    /**
     * Last element child.
     *
     * @returns Element.
     */
    get lastElementChild() {
        return this.children ? this.children[this.children.length - 1] || null : null;
    }
    /**
     * Last element child.
     *
     * @returns Element.
     */
    get childElementCount() {
        return this.children.length;
    }
    /**
     * Returns slot.
     *
     * @returns Slot.
     */
    get slot() {
        return this.getAttributeNS(null, 'slot') || '';
    }
    /**
     * Returns slot.
     *
     * @param slot Slot.
     */
    set slot(title) {
        this.setAttributeNS(null, 'slot', title);
    }
    /**
     * Returns inner HTML and optionally the content of shadow roots.
     *
     * This is a feature implemented in Chromium, but not supported by Mozilla yet.
     *
     * @see https://web.dev/declarative-shadow-dom/
     * @see https://chromestatus.com/feature/5191745052606464
     * @param [options] Options.
     * @param [options.includeShadowRoots] Set to "true" to include shadow roots.
     * @returns HTML.
     */
    getInnerHTML(options) {
        const xmlSerializer = new XMLSerializer_1.default();
        let xml = '';
        for (const node of this.childNodes) {
            xml += xmlSerializer.serializeToString(node, options);
        }
        return xml;
    }
    /**
     * Clones a node.
     *
     * @override
     * @param [deep=false] "true" to clone deep.
     * @returns Cloned node.
     */
    cloneNode(deep = false) {
        const clone = super.cloneNode(deep);
        for (const key of Object.keys(this._attributes)) {
            const attr = Object.assign(new Attr_1.default(), this._attributes[key]);
            attr.ownerElement = clone;
            clone._attributes[key] = attr;
        }
        if (deep) {
            for (const node of clone.childNodes) {
                if (node.nodeType === Node_1.default.ELEMENT_NODE) {
                    clone.children.push(node);
                }
            }
        }
        clone.tagName = this.tagName;
        clone.scrollLeft = this.scrollLeft;
        clone.scrollTop = this.scrollTop;
        clone.namespaceURI = this.namespaceURI;
        return clone;
    }
    /**
     * Append a child node to childNodes.
     *
     * @override
     * @param  node Node to append.
     * @returns Appended node.
     */
    appendChild(node) {
        // If the type is DocumentFragment, then the child nodes of if it should be moved instead of the actual node.
        // See: https://developer.mozilla.org/en-US/docs/Web/API/DocumentFragment
        if (node.nodeType !== Node_1.default.DOCUMENT_FRAGMENT_NODE) {
            if (node.parentNode && node.parentNode['children']) {
                const index = node.parentNode['children'].indexOf(node);
                if (index !== -1) {
                    node.parentNode['children'].splice(index, 1);
                }
            }
            if (node !== this && node.nodeType === Node_1.default.ELEMENT_NODE) {
                this.children.push(node);
            }
        }
        return super.appendChild(node);
    }
    /**
     * Remove Child element from childNodes array.
     *
     * @override
     * @param node Node to remove.
     */
    removeChild(node) {
        if (node.nodeType === Node_1.default.ELEMENT_NODE) {
            const index = this.children.indexOf(node);
            if (index !== -1) {
                this.children.splice(index, 1);
            }
        }
        return super.removeChild(node);
    }
    /**
     * Removes the node from its parent.
     */
    remove() {
        ChildNodeUtility_1.default.remove(this);
    }
    /**
     * Inserts a node before another.
     *
     * @override
     * @param newNode Node to insert.
     * @param [referenceNode] Node to insert before.
     * @returns Inserted node.
     */
    insertBefore(newNode, referenceNode) {
        const returnValue = super.insertBefore(newNode, referenceNode);
        // If the type is DocumentFragment, then the child nodes of if it should be moved instead of the actual node.
        // See: https://developer.mozilla.org/en-US/docs/Web/API/DocumentFragment
        if (newNode.nodeType !== Node_1.default.DOCUMENT_FRAGMENT_NODE) {
            if (newNode.parentNode && newNode.parentNode['children']) {
                const index = newNode.parentNode['children'].indexOf(newNode);
                if (index !== -1) {
                    newNode.parentNode['children'].splice(index, 1);
                }
            }
            this.children.length = 0;
            for (const node of this.childNodes) {
                if (node.nodeType === Node_1.default.ELEMENT_NODE) {
                    this.children.push(node);
                }
            }
        }
        return returnValue;
    }
    /**
     * The Node.replaceWith() method replaces this Node in the children list of its parent with a set of Node or DOMString objects.
     *
     * @param nodes List of Node or DOMString.
     */
    replaceWith(...nodes) {
        ChildNodeUtility_1.default.replaceWith(this, ...nodes);
    }
    /**
     * Inserts a set of Node or DOMString objects in the children list of this ChildNode's parent, just before this ChildNode. DOMString objects are inserted as equivalent Text nodes.
     *
     * @param nodes List of Node or DOMString.
     */
    before(...nodes) {
        ChildNodeUtility_1.default.before(this, ...nodes);
    }
    /**
     * Inserts a set of Node or DOMString objects in the children list of this ChildNode's parent, just after this ChildNode. DOMString objects are inserted as equivalent Text nodes.
     *
     * @param nodes List of Node or DOMString.
     */
    after(...nodes) {
        ChildNodeUtility_1.default.after(this, ...nodes);
    }
    /**
     * Inserts a set of Node objects or DOMString objects after the last child of the ParentNode. DOMString objects are inserted as equivalent Text nodes.
     *
     * @param nodes List of Node or DOMString.
     */
    append(...nodes) {
        ParentNodeUtility_1.default.append(this, ...nodes);
    }
    /**
     * Inserts a set of Node objects or DOMString objects before the first child of the ParentNode. DOMString objects are inserted as equivalent Text nodes.
     *
     * @param nodes List of Node or DOMString.
     */
    prepend(...nodes) {
        ParentNodeUtility_1.default.prepend(this, ...nodes);
    }
    /**
     * Replaces the existing children of a node with a specified new set of children.
     *
     * @param nodes List of Node or DOMString.
     */
    replaceChildren(...nodes) {
        ParentNodeUtility_1.default.replaceChildren(this, ...nodes);
    }
    /**
     * Inserts a node to the given position.
     *
     * @param position Position to insert element.
     * @param element Node to insert.
     * @returns Inserted node or null if couldn't insert.
     */
    insertAdjacentElement(position, element) {
        if (position === 'beforebegin') {
            if (!this.parentElement) {
                return null;
            }
            this.parentElement.insertBefore(element, this);
        }
        else if (position === 'afterbegin') {
            this.insertBefore(element, this.firstChild);
        }
        else if (position === 'beforeend') {
            this.appendChild(element);
        }
        else if (position === 'afterend') {
            if (!this.parentElement) {
                return null;
            }
            this.parentElement.insertBefore(element, this.nextSibling);
        }
        return element;
    }
    /**
     * Inserts an HTML string to the given position.
     *
     * @param position Position to insert text.
     * @param text HTML string to insert.
     */
    insertAdjacentHTML(position, text) {
        for (const node of XMLParser_1.default.parse(this.ownerDocument, text).childNodes.slice()) {
            this.insertAdjacentElement(position, node);
        }
    }
    /**
     * Inserts text to the given position.
     *
     * @param position Position to insert text.
     * @param text String to insert.
     */
    insertAdjacentText(position, text) {
        if (!text) {
            return;
        }
        const textNode = this.ownerDocument.createTextNode(text);
        this.insertAdjacentElement(position, textNode);
    }
    /**
     * Sets an attribute.
     *
     * @param name Name.
     * @param value Value.
     */
    setAttribute(name, value) {
        const attribute = this.ownerDocument.createAttributeNS(null, name);
        attribute.value = String(value);
        this.setAttributeNode(attribute);
    }
    /**
     * Sets a namespace attribute.
     *
     * @param namespaceURI Namespace URI.
     * @param name Name.
     * @param value Value.
     */
    setAttributeNS(namespaceURI, name, value) {
        const attribute = this.ownerDocument.createAttributeNS(namespaceURI, name);
        attribute.value = String(value);
        this.setAttributeNode(attribute);
    }
    /**
     * Returns attribute names.
     *
     * @returns Attribute names.
     */
    getAttributeNames() {
        return Object.keys(this._attributes);
    }
    /**
     * Returns attribute value.
     *
     * @param name Name.
     */
    getAttribute(name) {
        const attribute = this.getAttributeNode(name);
        if (attribute) {
            return attribute.value;
        }
        return null;
    }
    /**
     * Returns namespace attribute value.
     *
     * @param namespace Namespace URI.
     * @param localName Local name.
     */
    getAttributeNS(namespace, localName) {
        const attribute = this.getAttributeNodeNS(namespace, localName);
        if (attribute) {
            return attribute.value;
        }
        return null;
    }
    /**
     * Returns a boolean value indicating whether the specified element has the attribute or not.
     *
     * @param name Attribute name.
     * @returns True if attribute exists, false otherwise.
     */
    hasAttribute(name) {
        return !!this.getAttributeNode(name);
    }
    /**
     * Returns a boolean value indicating whether the specified element has the namespace attribute or not.
     *
     * @param namespace Namespace URI.
     * @param localName Local name.
     * @returns True if attribute exists, false otherwise.
     */
    hasAttributeNS(namespace, localName) {
        for (const name of Object.keys(this._attributes)) {
            const attribute = this._attributes[name];
            if (attribute.namespaceURI === namespace && attribute.localName === localName) {
                return true;
            }
        }
        return false;
    }
    /**
     * Returns "true" if the element has attributes.
     *
     * @returns "true" if the element has attributes.
     */
    hasAttributes() {
        return Object.keys(this._attributes).length > 0;
    }
    /**
     * Removes an attribute.
     *
     * @param name Name.
     */
    removeAttribute(name) {
        const attribute = this._attributes[this._getAttributeName(name)];
        if (attribute) {
            this.removeAttributeNode(attribute);
        }
    }
    /**
     * Removes a namespace attribute.
     *
     * @param namespace Namespace URI.
     * @param localName Local name.
     */
    removeAttributeNS(namespace, localName) {
        for (const name of Object.keys(this._attributes)) {
            const attribute = this._attributes[name];
            if (attribute.namespaceURI === namespace && attribute.localName === localName) {
                this.removeAttribute(attribute.name);
            }
        }
    }
    /**
     * Attaches a shadow root.
     *
     * @param _shadowRootInit Shadow root init.
     * @param shadowRootInit
     * @param shadowRootInit.mode
     * @returns Shadow root.
     */
    attachShadow(shadowRootInit) {
        if (this._shadowRoot) {
            throw new DOMException_1.default('Shadow root has already been attached.');
        }
        this._shadowRoot = new ShadowRoot_1.default();
        this._shadowRoot.ownerDocument = this.ownerDocument;
        this._shadowRoot.host = this;
        this._shadowRoot.mode = shadowRootInit.mode;
        this._shadowRoot._connectToNode(this);
        if (this._shadowRoot.mode === 'open') {
            this.shadowRoot = this._shadowRoot;
        }
        return this._shadowRoot;
    }
    /**
     * Converts to string.
     *
     * @returns String.
     */
    toString() {
        return this.outerHTML;
    }
    /**
     * Returns the size of an element and its position relative to the viewport.
     *
     * @returns DOM rect.
     */
    getBoundingClientRect() {
        // TODO: Not full implementation
        return new DOMRect_1.default();
    }
    /**
     * Returns a collection of DOMRect objects that indicate the bounding rectangles for each CSS border box in a client.
     *
     * @see https://developer.mozilla.org/en-US/docs/Web/API/Element/getClientRects
     * @returns DOM rect list.
     */
    getClientRects() {
        // TODO: Not full implementation
        return DOMRectListFactory_1.default.create([this.getBoundingClientRect()]);
    }
    /**
     * The matches() method checks to see if the Element would be selected by the provided selectorString.
     *
     * @param selector Selector.
     * @returns "true" if matching.
     */
    matches(selector) {
        for (const part of selector.split(',')) {
            if (new SelectorItem_1.default(part.trim()).match(this)) {
                return true;
            }
        }
        return false;
    }
    /**
     * Traverses the Element and its parents (heading toward the document root) until it finds a node that matches the provided selector string.
     *
     * @param selector Selector.
     * @returns Closest matching element.
     */
    closest(selector) {
        let rootElement = this.ownerDocument.documentElement;
        if (!this.isConnected) {
            rootElement = this;
            while (rootElement.parentNode) {
                rootElement = rootElement.parentNode;
            }
        }
        const elements = rootElement.querySelectorAll(selector);
        // eslint-disable-next-line
        let parent = this;
        while (parent) {
            if (elements.includes(parent)) {
                return parent;
            }
            parent = parent.parentElement;
        }
        // QuerySelectorAll() will not match the element it is looking in when searched for
        // Therefore we need to check if it matches the root
        if (rootElement.matches(selector)) {
            return rootElement;
        }
        return null;
    }
    /**
     * Query CSS selector to find matching nodes.
     *
     * @param selector CSS selector.
     * @returns Matching elements.
     */
    querySelectorAll(selector) {
        return QuerySelector_1.default.querySelectorAll(this, selector);
    }
    /**
     * Query CSS Selector to find matching node.
     *
     * @param selector CSS selector.
     * @returns Matching element.
     */
    querySelector(selector) {
        return QuerySelector_1.default.querySelector(this, selector);
    }
    /**
     * Returns an elements by class name.
     *
     * @param className Tag name.
     * @returns Matching element.
     */
    getElementsByClassName(className) {
        return ParentNodeUtility_1.default.getElementsByClassName(this, className);
    }
    /**
     * Returns an elements by tag name.
     *
     * @param tagName Tag name.
     * @returns Matching element.
     */
    getElementsByTagName(tagName) {
        return ParentNodeUtility_1.default.getElementsByTagName(this, tagName);
    }
    /**
     * Returns an elements by tag name and namespace.
     *
     * @param namespaceURI Namespace URI.
     * @param tagName Tag name.
     * @returns Matching element.
     */
    getElementsByTagNameNS(namespaceURI, tagName) {
        return ParentNodeUtility_1.default.getElementsByTagNameNS(this, namespaceURI, tagName);
    }
    /**
     * The setAttributeNode() method adds a new Attr node to the specified element.
     *
     * @param attribute Attribute.
     * @returns Replaced attribute.
     */
    setAttributeNode(attribute) {
        const name = this._getAttributeName(attribute.name);
        const replacedAttribute = this._attributes[name];
        const oldValue = replacedAttribute ? replacedAttribute.value : null;
        attribute.name = name;
        attribute.ownerElement = this;
        attribute.ownerDocument = this.ownerDocument;
        this._attributes[name] = attribute;
        this._updateDomListIndices();
        if (this.attributeChangedCallback &&
            this.constructor._observedAttributes &&
            this.constructor._observedAttributes.includes(name)) {
            this.attributeChangedCallback(name, oldValue, attribute.value);
        }
        // MutationObserver
        if (this._observers.length > 0) {
            for (const observer of this._observers) {
                if (observer.options.attributes &&
                    (!observer.options.attributeFilter || observer.options.attributeFilter.includes(name))) {
                    const record = new MutationRecord_1.default();
                    record.target = this;
                    record.type = MutationTypeEnum_1.default.attributes;
                    record.attributeName = name;
                    record.oldValue = observer.options.attributeOldValue ? oldValue : null;
                    observer.callback([record]);
                }
            }
        }
        return replacedAttribute || null;
    }
    /**
     * The setAttributeNodeNS() method adds a new Attr node to the specified element.
     *
     * @param attribute Attribute.
     * @returns Replaced attribute.
     */
    setAttributeNodeNS(attribute) {
        return this.setAttributeNode(attribute);
    }
    /**
     * Returns an Attr node.
     *
     * @param name Name.
     * @returns Replaced attribute.
     */
    getAttributeNode(name) {
        return this._attributes[this._getAttributeName(name)] || null;
    }
    /**
     * Returns a namespaced Attr node.
     *
     * @param namespace Namespace.
     * @param name Name.
     * @returns Replaced attribute.
     */
    getAttributeNodeNS(namespace, name) {
        const attributeName = this._getAttributeName(name);
        if (this._attributes[attributeName] &&
            this._attributes[attributeName].namespaceURI === namespace &&
            this._attributes[attributeName].localName === attributeName) {
            return this._attributes[attributeName];
        }
        for (const name of Object.keys(this._attributes)) {
            const attribute = this._attributes[name];
            if (attribute.namespaceURI === namespace && attribute.localName === attributeName) {
                return attribute;
            }
        }
        return null;
    }
    /**
     * Removes an Attr node.
     *
     * @param attribute Attribute.
     */
    removeAttributeNode(attribute) {
        delete this._attributes[attribute.name];
        this._updateDomListIndices();
        if (this.attributeChangedCallback &&
            this.constructor._observedAttributes &&
            this.constructor._observedAttributes.includes(attribute.name)) {
            this.attributeChangedCallback(attribute.name, attribute.value, null);
        }
        // MutationObserver
        if (this._observers.length > 0) {
            for (const observer of this._observers) {
                if (observer.options.attributes &&
                    (!observer.options.attributeFilter ||
                        observer.options.attributeFilter.includes(attribute.name))) {
                    const record = new MutationRecord_1.default();
                    record.target = this;
                    record.type = MutationTypeEnum_1.default.attributes;
                    record.attributeName = attribute.name;
                    record.oldValue = observer.options.attributeOldValue ? attribute.value : null;
                    observer.callback([record]);
                }
            }
        }
    }
    /**
     * Removes an Attr node.
     *
     * @param attribute Attribute.
     */
    removeAttributeNodeNS(attribute) {
        this.removeAttributeNode(attribute);
    }
    /**
     * Scrolls to a particular set of coordinates.
     *
     * @param x X position or options object.
     * @param y Y position.
     */
    scroll(x, y) {
        if (typeof x === 'object') {
            if (x.behavior === 'smooth') {
                this.ownerDocument.defaultView.setTimeout(() => {
                    if (x.top !== undefined) {
                        this.scrollTop = x.top;
                    }
                    if (x.left !== undefined) {
                        this.scrollLeft = x.left;
                    }
                });
            }
            else {
                if (x.top !== undefined) {
                    this.scrollTop = x.top;
                }
                if (x.left !== undefined) {
                    this.scrollLeft = x.left;
                }
            }
        }
        else if (x !== undefined && y !== undefined) {
            this.scrollLeft = x;
            this.scrollTop = y;
        }
    }
    /**
     * Scrolls to a particular set of coordinates.
     *
     * @param x X position or options object.
     * @param y Y position.
     */
    scrollTo(x, y) {
        this.scroll(x, y);
    }
    /**
     * Returns attribute name.
     *
     * @param name Name.
     * @returns Attribute name based on namespace.
     */
    _getAttributeName(name) {
        if (this.namespaceURI === NamespaceURI_1.default.svg) {
            return name;
        }
        return name.toLowerCase();
    }
    /**
     * Updates DOM list indices.
     */
    _updateDomListIndices() {
        if (this._classList) {
            this._classList._updateIndices();
        }
    }
}
exports.default = Element;
//# sourceMappingURL=Element.js.map