"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const Element_1 = __importDefault(require("../element/Element"));
const CSSStyleDeclaration_1 = __importDefault(require("../../css/CSSStyleDeclaration"));
const FocusEvent_1 = __importDefault(require("../../event/events/FocusEvent"));
const PointerEvent_1 = __importDefault(require("../../event/events/PointerEvent"));
const Node_1 = __importDefault(require("../node/Node"));
const DatasetUtility_1 = __importDefault(require("./DatasetUtility"));
/**
 * HTML Element.
 *
 * Reference:
 * https://developer.mozilla.org/en-US/docs/Web/API/HTMLElement.
 */
class HTMLElement extends Element_1.default {
    constructor() {
        super(...arguments);
        this.accessKey = '';
        this.accessKeyLabel = '';
        this.contentEditable = 'inherit';
        this.isContentEditable = false;
        this.offsetHeight = 0;
        this.offsetWidth = 0;
        this.offsetLeft = 0;
        this.offsetTop = 0;
        this.clientHeight = 0;
        this.clientWidth = 0;
        this._style = null;
        this._dataset = null;
    }
    /**
     * Returns tab index.
     *
     * @returns Tab index.
     */
    get tabIndex() {
        const tabIndex = this.getAttributeNS(null, 'tabindex');
        return tabIndex !== null ? Number(tabIndex) : -1;
    }
    /**
     * Returns tab index.
     *
     * @param tabIndex Tab index.
     */
    set tabIndex(tabIndex) {
        if (tabIndex === -1) {
            this.removeAttributeNS(null, 'tabindex');
        }
        else {
            this.setAttributeNS(null, 'tabindex', String(tabIndex));
        }
    }
    /**
     * Returns inner text, which is the rendered appearance of text.
     *
     * @returns Inner text.
     */
    get innerText() {
        let result = '';
        for (const childNode of this.childNodes) {
            if (childNode instanceof HTMLElement) {
                if (childNode.tagName !== 'SCRIPT' && childNode.tagName !== 'STYLE') {
                    result += childNode.innerText;
                }
            }
            else if (childNode.nodeType === Node_1.default.ELEMENT_NODE ||
                childNode.nodeType === Node_1.default.TEXT_NODE) {
                result += childNode.textContent;
            }
        }
        return result;
    }
    /**
     * Sets the inner text, which is the rendered appearance of text.
     *
     * @param innerText Inner text.
     */
    set innerText(innerText) {
        this.textContent = innerText;
    }
    /**
     * Returns style.
     *
     * @returns Style.
     */
    get style() {
        if (!this._style) {
            this._style = new CSSStyleDeclaration_1.default(this._attributes);
        }
        return this._style;
    }
    /**
     * Returns data set.
     *
     * @returns Data set.
     */
    get dataset() {
        if (this._dataset) {
            return this._dataset;
        }
        const dataset = {};
        const attributes = this._attributes;
        for (const name of Object.keys(attributes)) {
            if (name.startsWith('data-')) {
                const key = DatasetUtility_1.default.kebabToCamelCase(name.replace('data-', ''));
                dataset[key] = attributes[name].value;
            }
        }
        // Documentation for Proxy:
        // https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Proxy
        this._dataset = new Proxy(dataset, {
            get: (dataset, key) => {
                const name = 'data-' + DatasetUtility_1.default.camelCaseToKebab(key);
                if (this._attributes[name]) {
                    dataset[key] = this._attributes[name].value;
                    return this._attributes[name].value;
                }
                if (dataset[key] !== undefined) {
                    delete dataset[key];
                }
                return undefined;
            },
            set: (dataset, key, value) => {
                this.setAttribute('data-' + DatasetUtility_1.default.camelCaseToKebab(key), value);
                dataset[key] = value;
                return true;
            },
            deleteProperty: (dataset, key) => {
                const name = 'data-' + DatasetUtility_1.default.camelCaseToKebab(key);
                const exists = !!attributes[name];
                delete attributes[name];
                delete dataset[key];
                return exists;
            },
            ownKeys: (dataset) => {
                // According to Mozilla we have to update the dataset object (target) to contain the same keys as what we return:
                // https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Proxy/Proxy/ownKeys
                // "The result List must contain the keys of all non-configurable own properties of the target object."
                const keys = [];
                const deleteKeys = [];
                for (const name of Object.keys(attributes)) {
                    if (name.startsWith('data-')) {
                        const key = DatasetUtility_1.default.kebabToCamelCase(name.replace('data-', ''));
                        keys.push(key);
                        dataset[key] = attributes[name].value;
                        if (!dataset[key]) {
                            deleteKeys.push(key);
                        }
                    }
                }
                for (const key of deleteKeys) {
                    delete dataset[key];
                }
                return keys;
            },
            has: (_dataset, key) => {
                return !!attributes['data-' + DatasetUtility_1.default.camelCaseToKebab(key)];
            }
        });
        return this._dataset;
    }
    /**
     * Returns direction.
     *
     * @returns Direction.
     */
    get dir() {
        return this.getAttributeNS(null, 'dir') || '';
    }
    /**
     * Returns direction.
     *
     * @param direction Direction.
     */
    set dir(direction) {
        this.setAttributeNS(null, 'dir', direction);
    }
    /**
     * Returns hidden.
     *
     * @returns Hidden.
     */
    get hidden() {
        return this.getAttributeNS(null, 'hidden') !== null;
    }
    /**
     * Returns hidden.
     *
     * @param hidden Hidden.
     */
    set hidden(hidden) {
        if (!hidden) {
            this.removeAttributeNS(null, 'hidden');
        }
        else {
            this.setAttributeNS(null, 'hidden', '');
        }
    }
    /**
     * Returns language.
     *
     * @returns Language.
     */
    get lang() {
        return this.getAttributeNS(null, 'lang') || '';
    }
    /**
     * Returns language.
     *
     * @param language Language.
     */
    set lang(lang) {
        this.setAttributeNS(null, 'lang', lang);
    }
    /**
     * Returns title.
     *
     * @returns Title.
     */
    get title() {
        return this.getAttributeNS(null, 'title') || '';
    }
    /**
     * Returns title.
     *
     * @param title Title.
     */
    set title(title) {
        this.setAttributeNS(null, 'title', title);
    }
    /**
     * Triggers a click event.
     */
    click() {
        const event = new PointerEvent_1.default('click', {
            bubbles: true,
            composed: true
        });
        event.target = this;
        event.currentTarget = this;
        this.dispatchEvent(event);
    }
    /**
     * Triggers a blur event.
     */
    blur() {
        if (this.ownerDocument['_activeElement'] !== this || !this.isConnected) {
            return;
        }
        this.ownerDocument['_activeElement'] = null;
        for (const eventType of ['blur', 'focusout']) {
            const event = new FocusEvent_1.default(eventType, {
                bubbles: true,
                composed: true
            });
            event.target = this;
            event.currentTarget = this;
            this.dispatchEvent(event);
        }
    }
    /**
     * Triggers a focus event.
     */
    focus() {
        if (this.ownerDocument['_activeElement'] === this || !this.isConnected) {
            return;
        }
        if (this.ownerDocument['_activeElement'] !== null) {
            this.ownerDocument['_activeElement'].blur();
        }
        this.ownerDocument['_activeElement'] = this;
        for (const eventType of ['focus', 'focusin']) {
            const event = new FocusEvent_1.default(eventType, {
                bubbles: true,
                composed: true
            });
            event.target = this;
            event.currentTarget = this;
            this.dispatchEvent(event);
        }
    }
    /**
     * The setAttributeNode() method adds a new Attr node to the specified element.
     *
     * @override
     * @param attribute Attribute.
     * @returns Replaced attribute.
     */
    setAttributeNode(attribute) {
        const replacedAttribute = super.setAttributeNode(attribute);
        if (attribute.name === 'style' && this._style) {
            this._style.cssText = attribute.value;
        }
        return replacedAttribute;
    }
    /**
     * Removes an Attr node.
     *
     * @override
     * @param attribute Attribute.
     */
    removeAttributeNode(attribute) {
        super.removeAttributeNode(attribute);
        if (attribute.name === 'style' && this._style) {
            this._style.cssText = '';
        }
    }
    /**
     * Clones a node.
     *
     * @override
     * @param [deep=false] "true" to clone deep.
     * @returns Cloned node.
     */
    cloneNode(deep = false) {
        const clone = super.cloneNode(deep);
        clone.accessKey = this.accessKey;
        clone.accessKeyLabel = this.accessKeyLabel;
        clone.contentEditable = this.contentEditable;
        clone.isContentEditable = this.isContentEditable;
        if (this._style) {
            clone.style.cssText = this._style.cssText;
        }
        return clone;
    }
}
exports.default = HTMLElement;
//# sourceMappingURL=HTMLElement.js.map