"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const HTMLElement_1 = __importDefault(require("../html-element/HTMLElement"));
const XMLParser_1 = __importDefault(require("../../xml-parser/XMLParser"));
const XMLSerializer_1 = __importDefault(require("../../xml-serializer/XMLSerializer"));
/**
 * HTML Template Element.
 *
 * Reference:
 * https://developer.mozilla.org/en-US/docs/Web/API/HTMLTemplateElement.
 */
class HTMLTemplateElement extends HTMLElement_1.default {
    constructor() {
        super(...arguments);
        this.content = this.ownerDocument.createDocumentFragment();
    }
    /**
     * @override
     */
    get innerHTML() {
        return this.getInnerHTML();
    }
    /**
     * @override
     */
    set innerHTML(html) {
        for (const child of this.content.childNodes.slice()) {
            this.content.removeChild(child);
        }
        for (const node of XMLParser_1.default.parse(this.ownerDocument, html).childNodes.slice()) {
            this.content.appendChild(node);
        }
    }
    /**
     * @override
     */
    get previousSibling() {
        return null;
    }
    /**
     * @override
     */
    get nextSibling() {
        return null;
    }
    /**
     * @override
     */
    get firstChild() {
        return this.content.firstChild;
    }
    /**
     * @override
     */
    get lastChild() {
        return this.content.lastChild;
    }
    /**
     * @override
     */
    getInnerHTML(options) {
        const xmlSerializer = new XMLSerializer_1.default();
        let xml = '';
        for (const node of this.content.childNodes) {
            xml += xmlSerializer.serializeToString(node, options);
        }
        return xml;
    }
    /**
     * @override
     */
    appendChild(node) {
        return this.content.appendChild(node);
    }
    /**
     * @override
     */
    removeChild(node) {
        return this.content.removeChild(node);
    }
    /**
     * @override
     */
    insertBefore(newNode, referenceNode) {
        return this.content.insertBefore(newNode, referenceNode);
    }
    /**
     * @override
     */
    replaceChild(newChild, oldChild) {
        return this.content.replaceChild(newChild, oldChild);
    }
    /**
     * @override
     */
    cloneNode(deep = false) {
        const clone = super.cloneNode(deep);
        clone.content = this.content.cloneNode(deep);
        return clone;
    }
}
exports.default = HTMLTemplateElement;
//# sourceMappingURL=HTMLTemplateElement.js.map