"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const Node_1 = __importDefault(require("../nodes/node/Node"));
const VoidElements_1 = __importDefault(require("../config/VoidElements"));
const he_1 = require("he");
/**
 * Utility for converting an element to string.
 */
class XMLSerializer {
    /**
     * Renders an element as HTML.
     *
     * @param root Root element.
     * @param [options] Options.
     * @param [options.includeShadowRoots] Set to "true" to include shadow roots.
     * @returns Result.
     */
    serializeToString(root, options) {
        switch (root.nodeType) {
            case Node_1.default.ELEMENT_NODE:
                const element = root;
                const tagName = element.tagName.toLowerCase();
                if (VoidElements_1.default.includes(tagName)) {
                    return `<${tagName}${this._getAttributes(element)}>`;
                }
                const childNodes = element.tagName === 'TEMPLATE'
                    ? root.content.childNodes
                    : root.childNodes;
                let innerHTML = '';
                for (const node of childNodes) {
                    innerHTML += this.serializeToString(node, options);
                }
                if (options?.includeShadowRoots && element.shadowRoot) {
                    innerHTML += `<template shadowroot="${element.shadowRoot.mode}">`;
                    for (const node of element.shadowRoot.childNodes) {
                        innerHTML += this.serializeToString(node, options);
                    }
                    innerHTML += '</template>';
                }
                return `<${tagName}${this._getAttributes(element)}>${innerHTML}</${tagName}>`;
            case Node_1.default.DOCUMENT_FRAGMENT_NODE:
            case Node_1.default.DOCUMENT_NODE:
                let html = '';
                for (const node of root.childNodes) {
                    html += this.serializeToString(node, options);
                }
                return html;
            case Node_1.default.COMMENT_NODE:
                return `<!--${root.textContent}-->`;
            case Node_1.default.TEXT_NODE:
                return root['textContent'];
            case Node_1.default.DOCUMENT_TYPE_NODE:
                const doctype = root;
                const identifier = doctype.publicId ? ' PUBLIC' : doctype.systemId ? ' SYSTEM' : '';
                const publicId = doctype.publicId ? ` "${doctype.publicId}"` : '';
                const systemId = doctype.systemId ? ` "${doctype.systemId}"` : '';
                return `<!DOCTYPE ${doctype.name}${identifier}${publicId}${systemId}>`;
        }
        return '';
    }
    /**
     * Returns attributes as a string.
     *
     * @param element Element.
     * @returns Attributes.
     */
    _getAttributes(element) {
        let attributeString = '';
        if (!element._attributes.is && element._isValue) {
            attributeString += ' is="' + (0, he_1.escape)(element._isValue) + '"';
        }
        for (const attribute of Object.values(element._attributes)) {
            if (attribute.value !== null) {
                attributeString += ' ' + attribute.name + '="' + (0, he_1.escape)(attribute.value) + '"';
            }
        }
        return attributeString;
    }
}
exports.default = XMLSerializer;
//# sourceMappingURL=XMLSerializer.js.map