var heimdall = (function () {
// All Credit for this goes to the Ember.js Core Team

// This exists because `Object.create(null)` is absurdly slow compared
// to `new EmptyObject()`. In either case, you want a null prototype
// when you're treating the object instances as arbitrary dictionaries
// and don't want your keys colliding with build-in methods on the
// default object prototype.

var proto = Object.create(null, {
  // without this, we will always still end up with (new
  // EmptyObject()).constructor === Object
  constructor: {
    value: undefined,
    enumerable: false,
    writable: true
  }
});

function EmptyObject() {}
EmptyObject.prototype = proto;

var _typeof = typeof Symbol === "function" && typeof Symbol.iterator === "symbol" ? function (obj) {
  return typeof obj;
} : function (obj) {
  return obj && typeof Symbol === "function" && obj.constructor === Symbol ? "symbol" : typeof obj;
};

var createClass = function () {
  function defineProperties(target, props) {
    for (var i = 0; i < props.length; i++) {
      var descriptor = props[i];
      descriptor.enumerable = descriptor.enumerable || false;
      descriptor.configurable = true;
      if ("value" in descriptor) descriptor.writable = true;
      Object.defineProperty(target, descriptor.key, descriptor);
    }
  }

  return function (Constructor, protoProps, staticProps) {
    if (protoProps) defineProperties(Constructor.prototype, protoProps);
    if (staticProps) defineProperties(Constructor, staticProps);
    return Constructor;
  };
}();

var slicedToArray = function () {
  function sliceIterator(arr, i) {
    var _arr = [];
    var _n = true;
    var _d = false;
    var _e = undefined;

    try {
      for (var _i = arr[Symbol.iterator](), _s; !(_n = (_s = _i.next()).done); _n = true) {
        _arr.push(_s.value);

        if (i && _arr.length === i) break;
      }
    } catch (err) {
      _d = true;
      _e = err;
    } finally {
      try {
        if (!_n && _i["return"]) _i["return"]();
      } finally {
        if (_d) throw _e;
      }
    }

    return _arr;
  }

  return function (arr, i) {
    if (Array.isArray(arr)) {
      return arr;
    } else if (Symbol.iterator in Object(arr)) {
      return sliceIterator(arr, i);
    } else {
      throw new TypeError("Invalid attempt to destructure non-iterable instance");
    }
  };
}();

var UNDEFINED_KEY = Object.create(null);

var HashMap = function () {
  function HashMap(entries) {
    this._data = new EmptyObject();

    if (entries) {
      for (var i = 0; i < entries.length; i++) {
        this.data[entries[i][0]] = entries[i][1];
      }
    }
  }

  createClass(HashMap, [{
    key: 'forEach',
    value: function forEach(cb) {
      for (var key in this._data) {
        // skip undefined
        if (this._data[key] !== UNDEFINED_KEY) {
          cb(this._data[key], key);
        }
      }

      return this;
    }
  }, {
    key: 'has',
    value: function has(key) {
      return key in this._data && this._data[key] !== UNDEFINED_KEY;
    }
  }, {
    key: 'get',
    value: function get(key) {
      var val = this._data[key];

      return val === UNDEFINED_KEY ? undefined : val;
    }
  }, {
    key: 'set',
    value: function set(key, value) {
      this._data[key] = value;

      return this;
    }
  }, {
    key: 'delete',
    value: function _delete(key) {
      this._data[key] = UNDEFINED_KEY;
    }
  }]);
  return HashMap;
}();

var SMALL_ARRAY_LENGTH = 250;

var EventArray = function () {
  function EventArray() {
    var length = arguments.length <= 0 || arguments[0] === undefined ? SMALL_ARRAY_LENGTH : arguments[0];
    var initialData = arguments[1];

    this.init(length, initialData);
  }

  createClass(EventArray, [{
    key: "toJSON",
    value: function toJSON() {
      return this._data.slice(0, this.length);
    }
  }, {
    key: "init",
    value: function init() {
      var length = arguments.length <= 0 || arguments[0] === undefined ? SMALL_ARRAY_LENGTH : arguments[0];
      var initialData = arguments[1];

      this.length = 0;
      this._length = length;
      this._data = new Array(length);

      if (initialData) {
        if (initialData.length > length) {
          length = initialData.length;
          this._data.length = length;
          this._length = length;
        }

        for (var j = 0; j < initialData.length; j++) {
          this._data[j] = initialData[j];
          this.length++;
        }
      }
    }
  }, {
    key: "get",
    value: function get(index) {
      if (index >= 0 && index < this.length) {
        return this._data.slice(index, index + 4);
      }

      return undefined;
    }
  }, {
    key: "set",
    value: function set(index, value) {
      if (index > this.length) {
        throw new Error("Index is out of array bounds.");
      }

      if (index === this.length) {
        this.length++;
      }

      this._data[index] = value;
    }
  }, {
    key: "forEach",
    value: function forEach(cb) {
      for (var i = 0; i < this.length; i += 4) {
        cb(this._data.slice(i, i + 4), i);
      }
    }
  }, {
    key: "push",
    value: function push(op, name, time, data) {
      var index = this.length;
      this.length += 4;

      if (index >= this._length) {
        this._length *= 2;
        this._data.length = this._length;
      }

      this._data[index] = op;
      this._data[index + 1] = name;
      this._data[index + 2] = time;
      this._data[index + 3] = data;

      return index;
    }
  }, {
    key: "pop",
    value: function pop() {
      var index = --this.length;

      if (index < 0) {
        this.length = 0;
        return undefined;
      }

      return this._data[index];
    }
  }]);
  return EventArray;
}();

var SMALL_ARRAY_LENGTH$1 = 250;
var MAX_ARRAY_LENGTH = 1e6;

var FastIntArray = function () {
  function FastIntArray() {
    var length = arguments.length <= 0 || arguments[0] === undefined ? SMALL_ARRAY_LENGTH$1 : arguments[0];
    var initialData = arguments[1];

    this.init(length, initialData);
  }

  createClass(FastIntArray, [{
    key: "init",
    value: function init() {
      var length = arguments.length <= 0 || arguments[0] === undefined ? SMALL_ARRAY_LENGTH$1 : arguments[0];
      var initialData = arguments[1];

      this.length = 0;
      this._length = length;
      this._fill = 0;
      this._data = new Uint32Array(length);

      if (initialData) {
        if (initialData.length > length) {
          length = initialData.length;

          this.grow(length);
        }

        for (var j = 0; j < initialData.length; j++) {
          this._data[j] = initialData[j];
          this.length++;
        }
      }
    }
  }, {
    key: "toJSON",
    value: function toJSON() {
      return this._data.slice(0, this.length);
    }
  }, {
    key: "get",
    value: function get(index) {
      if (index >= 0 && index < this.length) {
        return this._data[index];
      }

      return undefined;
    }
  }, {
    key: "increment",
    value: function increment(index) {
      this._data[index]++;
    }

    /*
     Uint32Arrays have an immutable length. This method
     enables us to efficiently increase the length by
     any quantity.
     */

  }, {
    key: "grow",
    value: function grow(newLength) {
      var l = this._length;
      this._length = newLength;

      var data = this._data;
      var _d = this._data = new Uint32Array(newLength);

      _d.set(data);

      if (this._fill !== 0) {
        _d.fill(this._fill, l);
      }
    }
  }, {
    key: "claim",
    value: function claim(count) {
      this.length += count;
      while (this.length > this._length) {
        this.grow(this._length * 2);
      }
    }
  }, {
    key: "push",
    value: function push(int) {
      var index = this.length++;

      if (index === this._length) {
        this.grow(this._length * 2);
      }

      this._data[index] = int;
    }
  }]);
  return FastIntArray;
}();

var DEFAULT_STORE_SIZE = 1e3;
var DEFAULT_NAMESPACE_SIZE = 10;

// NULL_NUMBER is a number larger than the largest
// index we are capable of utilizing in the store.
// if an index is this number, we know that it is null.
var NULL_NUMBER = MAX_ARRAY_LENGTH + 1;
var LOB = (1 << 16) - 1;

var CounterStore = function () {
  function CounterStore() {
    var options = arguments.length <= 0 || arguments[0] === undefined ? {} : arguments[0];

    this.options = options;
    this.initialized = false;
    this._storeInitialized = false;
    this._store = null;
    this._namespaceCount = 0;
    this._config = null;
    this._cache = null;
    this._labelCache = null;
    this._nameCache = null;
  }

  createClass(CounterStore, [{
    key: 'toJSON',
    value: function toJSON() {
      return {
        _namespaceCount: this._namespaceCount,
        _config: this._config,
        _labelCache: this._labelCache,
        _nameCache: this._nameCache,
        _store: this._store
      };
    }
  }, {
    key: 'registerNamespace',
    value: function registerNamespace(name, labels) {
      this._initializeIfNeeded();

      var numCounters = labels.length;
      var counters = new Array(numCounters);
      var namespaceIndex = this._namespaceCount++;
      var bitNamespaceIndex = namespaceIndex << 16;

      // we also generate a map between the counters
      // and these labels so that we can reconstruct
      // a meaningful structure later.
      this._nameCache[namespaceIndex] = name;
      this._labelCache[name] = labels;

      // grow the existing config and cache to account
      // for the new namespace
      this._config.push(numCounters);

      if (this._cache !== null) {
        var cache = this._cache;

        this._cache = new Uint32Array(this._namespaceCount);
        this._cache.set(cache);
        this._cache[namespaceIndex] = NULL_NUMBER;
      }

      for (var i = 0; i < numCounters; i++) {
        counters[i] = bitNamespaceIndex + i;
      }

      return counters;
    }
  }, {
    key: '_initializeIfNeeded',
    value: function _initializeIfNeeded() {
      if (this.initialized === false) {
        this._config = new FastIntArray(this.options.namespaceAllocation || DEFAULT_NAMESPACE_SIZE);
        this._labelCache = new EmptyObject();
        this._nameCache = new EmptyObject();
        this.initialized = true;
      }
    }
  }, {
    key: 'restoreFromCache',
    value: function restoreFromCache(cache) {
      var stats = new EmptyObject();

      for (var i = 0; i < cache.length; i++) {
        if (cache[i] !== NULL_NUMBER) {
          var startIndex = cache[i];
          var namespace = this._nameCache[i];
          var counterCount = this._config.get(i);

          stats[namespace] = new EmptyObject();

          for (var j = 0; j < counterCount; j++) {
            var storeIndex = startIndex + j;
            var label = this._labelCache[namespace][j];

            stats[namespace][label] = this._store.get(storeIndex);
          }
        }
      }

      return stats;
    }
  }, {
    key: 'increment',
    value: function increment(counter) {
      var namespaceIndex = counter >> 16;
      var counterIndex = counter & LOB;

      if (this._cache === null) {
        this._initializeStoreIfNeeded();
        this._cache = new Uint32Array(this._namespaceCount).fill(NULL_NUMBER);
      }

      if (this._cache[namespaceIndex] === NULL_NUMBER) {
        var counterCount = this._config.get(namespaceIndex);

        this._cache[namespaceIndex] = this._store.length;
        this._store.claim(counterCount);
      }

      var storeIndex = this._cache[namespaceIndex] + counterIndex;
      this._store.increment(storeIndex);
    }
  }, {
    key: '_initializeStoreIfNeeded',
    value: function _initializeStoreIfNeeded() {
      if (this._storeInitialized === false) {
        this._store = new FastIntArray(this.options.storeSize || DEFAULT_STORE_SIZE);
        this._storeInitialized = true;
      }
    }
  }, {
    key: 'has',
    value: function has(name) {
      return this._labelCache && name in this._labelCache;
    }
  }, {
    key: 'cache',
    value: function cache() {
      var cache = this._cache;
      this._cache = null;

      return cache;
    }
  }], [{
    key: 'fromJSON',
    value: function fromJSON(json) {
      var store = new CounterStore();
      store._namespaceCount = json._namespaceCount;
      store._labelCache = json._labelCache;
      store._nameCache = json._nameCache;

      if (json._store) {
        store._store = new FastIntArray(json._store.length, json._store);
      }

      if (json._config) {
        store._config = new FastIntArray(json._config.length, json._config);
      }
    }
  }]);
  return CounterStore;
}();

// provides easily interceptable indirection.
var HeimdallSession = function () {
  function HeimdallSession(options) {
    this.init(options);
  }

  // separate from constructor mostly for testing purposes


  createClass(HeimdallSession, [{
    key: 'init',
    value: function init() {
      this.monitors = new CounterStore();
      this.configs = new HashMap();
      this.events = new EventArray(640000 * 4);
    }
  }]);
  return HeimdallSession;
}();

var now = void 0;
var format = void 0;
var ORIGIN_TIME = void 0;

// It turns out to be nicer for perf to bind than to close over the time method
// however, when testing we need to be able to stub the clock via the global
// so we use this boolean to determine whether we "bind" or use a wrapper function.
var freeGlobal = typeof window !== 'undefined' ? window : global;
var IS_TESTING = freeGlobal.IS_HEIMDALL_TEST_ENVIRONMENT;

if ((typeof performance === 'undefined' ? 'undefined' : _typeof(performance)) === 'object' && typeof performance.now === 'function') {
  now = IS_TESTING ? function now() {
    return performance.now();
  } : performance.now.bind(performance);
  format = 'milli';
} else if (typeof process !== 'undefined' && typeof process.hrtime === 'function') {
  now = IS_TESTING ? function now() {
    return process.hrtime();
  } : process.hrtime.bind(process);
  format = 'hrtime';
} else {
  ORIGIN_TIME = Date.now();
  now = Date.now.bind(Date);
  format = 'timestamp';
}

function normalizeTime(time) {
  var format = arguments.length <= 1 || arguments[1] === undefined ? format : arguments[1];

  switch (format) {
    case 'milli':
      return milliToNano(time);
    case 'hrtime':
      return timeFromHRTime(time);
    case 'timestamp':
      return milliToNano(time - ORIGIN_TIME);
    default:
      throw new Error('Unknown Format');
  }
}

function milliToNano(time) {
  return Math.floor(time * 1e6);
}

function timeFromHRTime(hrtime) {
  return hrtime[0] * 1e9 + hrtime[1];
}

var now$1 = now;

var OP_START = 0;
var OP_STOP = 1;
var OP_RESUME = 2;
var OP_ANNOTATE = 3;

var Heimdall = function () {
  function Heimdall(session) {
    if (arguments.length < 1) {
      session = new HeimdallSession();
    }

    this._session = session;
  }

  createClass(Heimdall, [{
    key: '_retrieveCounters',
    value: function _retrieveCounters() {
      return this._monitors.cache();
    }
  }, {
    key: 'start',
    value: function start(name) {
      return this._session.events.push(OP_START, name, now$1(), this._retrieveCounters());
    }
  }, {
    key: 'stop',
    value: function stop(token) {
      this._session.events.push(OP_STOP, token, now$1(), this._retrieveCounters());
    }
  }, {
    key: 'resume',
    value: function resume(token) {
      this._session.events.push(OP_RESUME, token, now$1(), this._retrieveCounters());
    }
  }, {
    key: 'annotate',
    value: function annotate(info) {
      // This has the side effect of making events heterogenous, as info is an object
      // while counters will always be `null` or an `Array`
      this._session.events.push(OP_ANNOTATE, NULL_NUMBER, NULL_NUMBER, info);
    }
  }, {
    key: 'registerMonitor',
    value: function registerMonitor(name) {
      if (name === 'own' || name === 'time') {
        throw new Error('Cannot register monitor at namespace "' + name + '".  "own" and "time" are reserved');
      }
      if (this._monitors.has(name)) {
        throw new Error('A monitor for "' + name + '" is already registered"');
      }

      for (var _len = arguments.length, keys = Array(_len > 1 ? _len - 1 : 0), _key = 1; _key < _len; _key++) {
        keys[_key - 1] = arguments[_key];
      }

      return this._monitors.registerNamespace(name, keys);
    }
  }, {
    key: 'increment',
    value: function increment(token) {
      this._session.monitors.increment(token);
    }
  }, {
    key: 'configFor',
    value: function configFor(name) {
      var config = this._session.configs.get(name);

      if (!config) {
        config = new EmptyObject();
        this._session.configs.set(name, config);
      }

      return config;
    }

    /*
      Ideally, this method should only be used for serializing
      session data for transfer. Heimdall-tree can load time
      data from this format or out of `getSessionData`.
     */

  }, {
    key: 'toJSON',
    value: function toJSON() {
      return {
        format: format,
        monitors: this._monitors.toJSON(),
        events: this._events.toJSON()
      };
    }
  }, {
    key: 'toString',
    value: function toString() {
      return JSON.stringify(this.toJSON());
    }
  }, {
    key: '_monitors',
    get: function get() {
      return this._session.monitors;
    }
  }, {
    key: '_events',
    get: function get() {
      return this._session.events;
    }
  }]);
  return Heimdall;
}();

var HeimdallNode = function () {
  function HeimdallNode(name, id) {
    this._id = id;
    this.parent = null;
    this.resumeNode = null;
    this.name = name;
    this.stopped = false;
    this.leaves = [];
    this.nodes = [];
    this.children = [];
  }

  createClass(HeimdallNode, [{
    key: 'stop',
    value: function stop() {
      if (this.stopped === true) {
        throw new Error('Cannot Stop node, already stopped!');
      }
      this.stopped = true;
    }
  }, {
    key: 'resume',
    value: function resume(resumeNode) {
      if (!this.stopped) {
        throw new Error('Cannot Resume node, already running!');
      }
      this.resumeNode = resumeNode;
      this.stopped = false;
    }
  }, {
    key: 'addLeaf',
    value: function addLeaf(leaf) {
      leaf.owner = this;
      this.leaves.push(leaf);
      this.children.push(leaf);
    }
  }, {
    key: 'addNode',
    value: function addNode(node) {
      if (node.parent) {
        throw new Error('Cannot set parent of node \'' + node.name + '\', node already has a parent!');
      }
      node.parent = this;
      node.resumeNode = this;
      this.nodes.push(node);
      this.children.push(node);
    }
  }, {
    key: 'visitPreOrder',
    value: function visitPreOrder(cb) {
      cb(this);

      for (var i = 0; i < this.nodes.length; i++) {
        this.nodes[i].visitPreOrder(cb);
      }
    }
  }, {
    key: 'visitPostOrder',
    value: function visitPostOrder(cb) {
      for (var i = 0; i < this.nodes.length; i++) {
        this.nodes[i].visitPostOrder(cb);
      }

      cb(this);
    }
  }, {
    key: 'forEachNode',
    value: function forEachNode(cb) {
      for (var i = 0; i < this.nodes.length; ++i) {
        cb(this.nodes[i]);
      }
    }
  }, {
    key: 'forEachLeaf',
    value: function forEachLeaf(cb) {
      for (var i = 0; i < this.leaves.length; ++i) {
        cb(this.leaves[i]);
      }
    }
  }, {
    key: 'forEachChild',
    value: function forEachChild(cb) {
      for (var i = 0; i < this.children.length; ++i) {
        cb(this.children[i]);
      }
    }
  }, {
    key: 'toJSON',
    value: function toJSON() {
      return {
        _id: this._id,
        name: this.name,
        leaves: this.leaves.map(function (leaf) {
          return leaf.toJSON();
        }),
        nodes: this.nodes.map(function (child) {
          return child._id;
        }),
        children: this.children.map(function (child) {
          return child._id;
        })
      };
    }
  }, {
    key: 'toJSONSubgraph',
    value: function toJSONSubgraph() {
      var nodes = [];

      this.visitPreOrder(function (node) {
        return nodes.push(node.toJSON());
      });

      return nodes;
    }
  }, {
    key: 'stats',
    get: function get() {
      var own = {
        selfTime: 0,
        duration: 0,
        startTime: this.leaves[0].startTime,
        endTime: this.leaves[this.leaves.length - 1].endTime
      };
      own.duration = own.endTime - own.startTime;

      var counters = [];
      var annotations = [];
      var stats = {
        self: own
      };

      this.forEachLeaf(function (leaf) {
        own.selfTime += leaf.selfTime;
        annotations.push(leaf.annotations);

        for (var namespace in leaf.counters) {
          var value = leaf.counters[namespace];

          if (!stats.hasOwnProperty(namespace)) {
            stats[namespace] = value;
          } else {
            for (var label in value) {
              stats[namespace][label] += value[label];
            }
          }
        }

        counters.push(leaf.counters);
      });

      return stats;
    }
  }, {
    key: 'isRoot',
    get: function get() {
      return this.parent === null;
    }
  }]);
  return HeimdallNode;
}();

var HeimdallLeaf = function () {
  function HeimdallLeaf() {
    // set on start
    this._id = null;
    this.owner = null;
    this.previousOp = null;
    this.startTime = 0;

    // set on annotate
    this.annotations = null;

    // set on stop
    this.nextOp = null;
    this.endTime = 0;
    this.counters = null;
    this.name = null;
  }

  createClass(HeimdallLeaf, [{
    key: "annotate",
    value: function annotate(annotation) {
      if (this.annotations === null) {
        this.annotations = [];
      }
      this.annotations.push(annotation);
    }
  }, {
    key: "start",
    value: function start(owner, previousOp, time) {
      this.owner = owner;
      this.previousOp = previousOp;
      this.startTime = time;
    }
  }, {
    key: "stop",
    value: function stop(nextOp, time, counters) {
      this.nextOp = nextOp;
      this.endTime = time;
      this.counters = counters;
      this._id = this.name = "[" + this.owner.name + "]#" + this.previousOp + ":" + nextOp;
    }
  }, {
    key: "toJSON",
    value: function toJSON() {
      return {
        _id: this._id,
        name: this.name,
        startTime: this.startTime,
        endTime: this.endTime,
        counters: this.counters,
        annotations: this.annotations
      };
    }
  }, {
    key: "selfTime",
    get: function get() {
      return this.endTime - this.startTime;
    }
  }, {
    key: "isStopped",
    get: function get() {
      return this.endTime !== 0;
    }
  }]);
  return HeimdallLeaf;
}();

/*
Example Event Timeline and tree reconstruction

As       Bs       Cs       Ce       Be       Ae
|--------|--------|--------|--------|--------|
   AB        BC       CC      CB        BA

Tree
A
 \
  B
   \
    C

Leafy Tree
A <- node
 |_AB <- leaf
 |_ B <- child node
 | |_ BC
 | |
 | |_C
 | | |_ CC
 | |
 | |_ CB
 |
 |_ BA

*/

function statsFromCounters(counterStore, counterCache) {
  if (!counterStore || !counterCache) {
    return null;
  }

  return counterStore.restoreFromCache(counterCache);
}

var HeimdallTree = function () {
  function HeimdallTree(heimdall) {
    this._heimdall = heimdall;
    this.root = null;
    this.format = heimdall && heimdall._timeFormat ? heimdall._timeFormat : format;
  }

  createClass(HeimdallTree, [{
    key: 'construct',
    value: function construct() {
      var events = this._heimdall._events;
      var currentLeaf = null;
      var root = new HeimdallNode('root', 1e9);
      var currentNode = root;
      var nodeMap = new HashMap();
      var node = void 0;
      var format = this.format;
      var counterStore = this._heimdall._monitors;

      this.root = root;

      events.forEach(function (_ref, i) {
        var _ref2 = slicedToArray(_ref, 4);

        var op = _ref2[0];
        var name = _ref2[1];
        var time = _ref2[2];
        var counters = _ref2[3];

        if (op !== OP_ANNOTATE) {
          time = normalizeTime(time, format);
          counters = statsFromCounters(counterStore, counters);
        }

        switch (op) {
          case OP_START:
            node = new HeimdallNode(name, i);
            nodeMap.set(i, node);
            currentNode.addNode(node);
            currentNode = node;

            if (currentLeaf) {
              currentLeaf.stop(name, time, counters);
            }
            currentLeaf = new HeimdallLeaf();
            currentLeaf.start(currentNode, name, time);
            currentNode.addLeaf(currentLeaf);
            break;

          case OP_STOP:
            node = nodeMap.get(name);

            if (name !== currentNode._id) {
              // potentially throw the correct error (already stopped)
              if (node) {
                node.stop();
              } else {
                throw new Error("Cannot Stop, Attempting to stop a non-existent node!");
              }
              throw new Error("Cannot Stop, Attempting to stop a node with an active child!");
            }

            currentNode.stop();
            currentNode = currentNode.resumeNode;

            currentLeaf.stop(node.name, time, counters);
            currentLeaf = new HeimdallLeaf();
            currentLeaf.start(currentNode, node.name, time);
            currentNode.addLeaf(currentLeaf);
            break;

          case OP_RESUME:
            node = nodeMap.get(name);
            node.resume(currentNode);
            currentNode = node;

            if (currentLeaf) {
              currentLeaf.stop(node.name, time, counters);
            }
            currentLeaf = new HeimdallLeaf();
            currentLeaf.start(currentNode, node.name, time);
            currentNode.addLeaf(currentLeaf);
            break;

          case OP_ANNOTATE:
            currentLeaf.annotate(counters);
            break;
          default:
            throw new Error('HeimdallTree encountered an unknown OpCode \'' + op + '\' during tree construction.');
        }
      });

      if (currentLeaf) {
        root.leaves.splice(root.leaves.indexOf(currentLeaf), 1);
        root.children.splice(root.children.indexOf(currentLeaf), 1);
        currentLeaf.owner = null;
        currentLeaf = null;
      }
    }
  }, {
    key: 'toJSON',
    value: function toJSON() {
      if (!this.root) {
        this.construct();
      }
      return { nodes: this.root.toJSONSubgraph() };
    }
  }, {
    key: 'visitPreOrder',
    value: function visitPreOrder(cb) {
      return this.root.visitPreOrder(cb);
    }
  }, {
    key: 'visitPostOrder',
    value: function visitPostOrder(cb) {
      return this.root.visitPostOrder(cb);
    }
  }, {
    key: 'path',


    // primarily a test helper, you can get this at any time
    // to get an array representing the path of open node names
    // from "root" to the last open node.
    get: function get() {
      var events = this._heimdall._events;
      var root = new HeimdallNode('root', 1e9);
      var currentNode = root;
      var nodeMap = new HashMap();
      var node = void 0;
      var top = void 0;
      var path = [];

      events.forEach(function (_ref3, i) {
        var _ref4 = slicedToArray(_ref3, 2);

        var op = _ref4[0];
        var name = _ref4[1];

        switch (op) {
          case OP_START:
            node = new HeimdallNode(name, i);
            nodeMap.set(i, node);
            currentNode.addNode(node);
            currentNode = node;
            break;

          case OP_STOP:
            node = nodeMap.get(name);

            if (name !== currentNode._id) {
              // potentially throw the correct error (already stopped)
              if (node) {
                node.stop();
              } else {
                throw new Error("Cannot Stop, Attempting to stop a non-existent node!");
              }
              throw new Error("Cannot Stop, Attempting to stop a node with an active child!");
            }

            currentNode.stop();
            currentNode = currentNode.resumeNode;
            break;

          case OP_RESUME:
            node = nodeMap.get(name);
            node.resume(currentNode);
            currentNode = node;
            break;

          default:
            throw new Error('HeimdallTree encountered an unknown OpCode \'' + op + '\' during path construction.');
        }
      });

      top = currentNode;

      while (top !== undefined && top !== root) {
        path.unshift(top.name);
        top = top.parent;
      }

      return path;
    }

    // primarily a test helper, you can get this at any time
    // to get an array representing the "stack" of open node names.

  }, {
    key: 'stack',
    get: function get() {
      var events = this._heimdall._events;
      var stack = [];
      var nodeMap = new HashMap();

      events.forEach(function (_ref5, i) {
        var _ref6 = slicedToArray(_ref5, 2);

        var op = _ref6[0];
        var name = _ref6[1];

        if (op === OP_START) {
          stack.push(name);
          nodeMap.set(i, name);
        } else if (op === OP_RESUME) {
          var n = nodeMap.get(name);
          stack.push(n);
        } else if (op === OP_STOP) {
          var _n = nodeMap.get(name);

          if (_n !== stack[stack.length - 1]) {
            throw new Error('Invalid Stack!');
          }

          stack.pop();
        }
      });

      return stack;
    }
  }], [{
    key: 'fromJSON',
    value: function fromJSON(json) {
      var events = json.events || [];
      var heimdall = {
        _timeFormat: json.format || format,
        _events: new EventArray(events.length, events),
        _monitors: CounterStore.fromJSON(json.monitors)
      };

      return new HeimdallTree(heimdall);
    }
  }]);
  return HeimdallTree;
}();

function setupSession(global) {

  // The name of the property encodes the session/node compatibilty version
  if (!global._heimdall_session_3) {
    global._heimdall_session_3 = new HeimdallSession();
  }
}

setupSession(self);

// browser equivalent of heimdall.js
self.Heimdall = Heimdall;
Heimdall.Session = HeimdallSession;
Heimdall.Tree = HeimdallTree;

var index = new Heimdall(self._heimdall_session_3);

return index;

}());