const byteStream = require('byte-stream')
const through = require('through2')
const pumpify = require('pumpify')

const errors = require('./errors')

const { linux: linuxConstants, parse } = require('filesystem-constants')
const {
  O_RDONLY,
  O_WRONLY,
  O_RDWR,
  O_CREAT,
  O_TRUNC,
  O_APPEND,
  O_SYNC,
  O_EXCL,
  O_ACCMODE
} = linuxConstants

class FileDescriptor {
  constructor (drive, path, stat, contentState, readable, writable, appending, creating) {
    this.drive = drive
    this.stat = stat
    this.path = path
    this.contentState = contentState

    this.readable = readable
    this.writable = writable
    this.creating = creating
    this.appending = appending

    this.position = null
    this.blockPosition = stat ? stat.offset : null
    this.blockOffset = 0

    this._err = null
    if (this.writable) {
      if (this.appending) {
        this._appendStream = this.drive.createReadStream(this.path)
        this.position = this.stat.size
      }
    }

    this._batcher = byteStream({ time: 100, limit: 4096 * 16 })

    this._range = null
  }

  read (buffer, offset, len, pos, cb) {
    if (!this.readable) return cb(new errors.BadFileDescriptor('File descriptor not open for reading.'))
    if (this.position !== null && this.position === pos) this._read(buffer, offset, len, cb)
    else this._seekAndRead(buffer, offset, len, pos, cb)
  }

  write (buffer, offset, len, pos, cb) {
    if (!this.writable) return cb(new errors.BadFileDescriptor('File descriptor not open for writing.'))
    if (!this.stat && !this.creating) {
      return process.nextTick(cb, new errors.BadFileDescriptor('File descriptor not open in create mode.'))
    }
    if (this.position !== null && pos !== this.position) {
      return process.nextTick(cb, new errors.BadFileDescriptor('Random-access writes are not currently supported.'))
    }
    if (this.appending && pos < this.stat.size) {
      return process.nextTick(cb, new errors.BadFileDescriptor('Position cannot be less than the file size when appending.'))
    }
    if (!this._writeStream && !this.appending && pos) {
      return process.nextTick(cb, new errors.BadFileDescriptor('Random-access writes are not currently supported.'))
    }

    const self = this

    if (!this._writeStream) {
      this._writeStream = createWriteStream(this.drive, this.stat, this.path)
      this.drive._writingFds.set(this.path, this)
      this._writeStream.on('error', cb)
    }

    // TODO: This is a temporary (bad) way of supporting appends.
    if (this._appendStream) {
      this.position = this.stat.size
      // pump does not support the `end` option.
      this._appendStream.pipe(this._writeStream, { end: false })

      this._appendStream.on('error', err => this._writeStream.destroy(err))
      this._writeStream.on('error', err => this._appendStream.destroy(err))

      return this._appendStream.on('end', doWrite)
    }

    return doWrite()

    function doWrite (err) {
      self._appendStream = null
      if (err) return cb(err)
      if (self._err) return cb(self._err)
      if (self._writeStream.destroyed) return cb(new errors.BadFileDescriptor('Write stream was destroyed.'))

      const slice = buffer.slice(offset, len)
      write(self._writeStream, slice, err => {
        if (err) return cb(err)
        self.position += slice.length
        self.stat.size += slice.length
        return cb(null, slice.length, buffer)
      })
    }

    function write (stream, data, cb) {
      if (stream.write(data) === false) {
        stream.once('drain', cb)
      } else {
        process.nextTick(cb)
      }
    }
  }

  close (cb) {
    // TODO: undownload initial range
    this.drive._writingFds.delete(this.path)
    if (this._writeStream) {
      if (this._writeStream.destroyed) {
        this._writeStream = null
      } else {
        return this._writeStream.end(err => {
          if (err) return cb(err)
          this._writeStream = null
          return cb(null)
        })
      }
    }
    if (this._range) {
      this.contentState.feed.undownload(this._range)
      this._range = null
    }
    process.nextTick(cb, null)
  }

  /**
   * Will currently request the next 16 blocks linearly.
   *
   * TODO: This behavior should be more customizable in the future.
   */
  _refreshDownload (start, cb) {
    const end = Math.min(this.stat.blocks + this.stat.offset, start + 16)

    if (this._range) {
      this.contentState.feed.undownload(this._range)
    }

    this._range = this.contentState.feed.download({ start, end, linear: true }, cb || noop)
  }

  _seekAndRead (buffer, offset, len, pos, cb) {
    const start = this.stat.offset
    const end = start + this.stat.blocks

    this.contentState.feed.seek(this.stat.byteOffset + pos, { start, end }, (err, blk, blockOffset) => {
      if (err) return cb(err)
      this.position = pos
      this.blockPosition = blk
      this.blockOffset = blockOffset

      this._refreshDownload(blk)
      this._read(buffer, offset, len, cb)
    })
  }

  _read (buffer, offset, len, cb) {
    const self = this

    let totalRead = 0
    let available = len
    readNextBlock()

    function readNextBlock () {
      self._readBlock(buffer, offset + totalRead, Math.max(available, 0), (err, bytesRead) => {
        if (err) return cb(err)
        if (!bytesRead) return cb(null, totalRead, buffer)

        totalRead += bytesRead
        available -= bytesRead

        if (available > 0) {
          return readNextBlock()
        }
        return cb(null, totalRead, buffer)
      })
    }
  }

  _readBlock (buffer, offset, len, cb) {
    const buf = buffer.slice(offset, offset + len)
    const blkOffset = this.blockOffset
    const blk = this.blockPosition

    if (this._range && (blk < this._range.start || blk > this._range.end)) {
      this._refreshDownload(blk)
    }

    if ((this.stat.offset + this.stat.blocks) <= blk || blk < this.stat.offset) {
      return process.nextTick(cb, null, 0, buffer)
    }

    this.contentState.feed.get(blk, (err, data) => {
      if (err) return cb(err)
      if (blkOffset) data = data.slice(blkOffset)

      data.copy(buf)
      const read = Math.min(data.length, buf.length)

      if (blk === this.blockPosition && blkOffset === this.blockOffset) {
        this.position += read
        if (read === data.length) {
          this.blockPosition++
          this.blockOffset = 0
        } else {
          this.blockOffset = blkOffset + read
        }
      }

      cb(null, read, buffer)
    })
  }
}

module.exports = function create (drive, name, flags, cb) {
  try {
    flags = parse(linuxConstants, flags)
  } catch (err) {
    return process.nextTick(cb, new errors.InvalidArgument(err.message))
  }

  const accmode = flags & O_ACCMODE
  const writable = !!(accmode & (O_WRONLY | O_RDWR))
  const readable = accmode === 0 || !!(accmode & O_RDWR)
  const appending = !!(flags & O_APPEND)
  const creating = !!(flags & O_CREAT)
  const canExist = !(flags & O_EXCL)

  drive.stat(name, { trie: true }, (err, st, trie) => {
    if (err && (err.errno !== 2)) return cb(err)
    if (st && !canExist) return cb(new errors.PathAlreadyExists(name))
    if (!st && (!writable || !creating)) return cb(new errors.FileNotFound(name))

    drive._getContent(trie, (err, contentState) => {
      if (err) return cb(err)
      const fd = new FileDescriptor(drive, name, st, contentState, readable, writable, appending, creating)
      if (creating) {
        drive.create(name, (err, st) => {
          if (err) return cb(err)
          fd.stat = st
          return cb(null, fd)
        })
      } else {
        return cb(null, fd)
      }
    })
  })
}

function createWriteStream (drive, opts, path) {
  const writeStream = drive.createWriteStream(path, opts)
  const batcher = byteStream({ time: 100, limit: 4096 * 16 })
  return pumpify(batcher, through.obj((chunk, enc, cb) => {
    cb(null, Buffer.concat(chunk))
  }), writeStream)
}

function noop () {}
