"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.CompositeMediaElement = void 0;
const relative_time_1 = require("../helpers/relative-time");
const logger_1 = require("../helpers/logger");
class CompositeMediaElement {
    constructor(mediaElements) {
        this.elements = [];
        this.playing = false;
        this.canvasMap = {};
        this._onPlay = [];
        this._onPause = [];
        this._onBuffering = [];
        logger_1.Logger.log('Composite media element', mediaElements);
        // Add all elements.
        this.elements = mediaElements;
        for (const el of mediaElements) {
            const canvasId = el.getCanvasId();
            this.canvasMap[canvasId] = this.canvasMap[canvasId] ? this.canvasMap[canvasId] : [];
            this.canvasMap[canvasId].push(el);
            // Attach events.
            el.addEventListener('play', () => {
                if (el === this.activeElement) {
                    logger_1.Logger.log('HTMLElement.play() response', {
                        paused: el.getRawElement().paused,
                        readState: el.getRawElement().readyState,
                    });
                    this._onPlay.forEach((fn) => fn(canvasId, el.getElementTime(), el));
                }
            });
            el.addEventListener('pause', () => {
                if (el === this.activeElement) {
                    logger_1.Logger.log('HTMLElement.pause() response');
                    this._onPause.forEach((fn) => fn(canvasId, el.getElementTime(), el));
                }
            });
            el.addEventListener('waiting', () => {
                if (el === this.activeElement) {
                    this._onBuffering.forEach((fn) => fn(canvasId, el.getElementTime(), el));
                }
            });
        }
        this.activeElement = mediaElements[0];
    }
    syncClock(time, _toCanvas) {
        logger_1.Logger.group('CompositeMediaElement.syncClock');
        logger_1.Logger.log(`syncClock: ${time}`);
        logger_1.Logger.log({
            fromTime: time,
            toTime: time,
            instance: this,
        });
        if (this.activeElement) {
            const toCanvas = _toCanvas || this.activeElement.getCanvasId();
            this.updateActiveElement(toCanvas, time, this.playing);
            const realTime = (0, relative_time_1.minusTime)(time, this.activeElement.source.start);
            this.activeElement.syncClock(realTime);
        }
        logger_1.Logger.groupEnd();
    }
    updateActiveElement(canvasId, time, play) {
        const newElement = this.findElementInRange(canvasId, time);
        if (this.activeElement && newElement && newElement !== this.activeElement) {
            logger_1.Logger.log(`CompositeMediaElement.updateActiveElement(canvasId: ${canvasId}, time: ${time})`, {
                canvasId: newElement ? newElement.source.canvasId : null,
                newElement,
            });
            // Moving track.
            // Stop the current track.
            this.activeElement.stop();
            // Set new current track.
            this.activeElement = newElement;
            if (play) {
                newElement.play(time);
            }
            return newElement;
        }
        return null;
    }
    onPlay(func) {
        this._onPlay.push(func);
    }
    onPause(func) {
        this._onPause.push(func);
    }
    onBuffering(func) {
        this._onBuffering.push(func);
    }
    findElementInRange(canvasId, time) {
        if (!this.canvasMap[canvasId]) {
            return undefined;
        }
        for (const el of this.canvasMap[canvasId].reverse()) {
            if (el.isWithinRange(time)) {
                return el;
            }
        }
        return undefined;
    }
    appendTo($element) {
        $element.append(this.elements.map((media) => media.getRawElement()));
    }
    load() {
        return __awaiter(this, void 0, void 0, function* () {
            yield Promise.all(this.elements.map((element) => element.load()));
        });
    }
    seekToMediaTime(annotationTime, _toCanvas) {
        return __awaiter(this, void 0, void 0, function* () {
            const prevActiveElement = this.activeElement;
            logger_1.Logger.groupCollapsed('Buffering');
            if (this.activeElement) {
                const toCanvas = _toCanvas || this.activeElement.getCanvasId();
                const newElement = this.updateActiveElement(toCanvas, annotationTime, false);
                const realTime = (0, relative_time_1.minusTime)(annotationTime, this.activeElement.source.start);
                let defer;
                const promise = new Promise((resolve) => (defer = resolve));
                if (this.playing) {
                    logger_1.Logger.log(`CompositeMediaElement.seekToMediaItem(${annotationTime})`);
                    yield this.activeElement.play(realTime).catch((e) => {
                        console.log('ERROR', e);
                        this.playing = false;
                    });
                    if (prevActiveElement !== this.activeElement && newElement) {
                        logger_1.Logger.log(`Active element changed...`);
                        if (newElement.isBuffering() || newElement.isPaused()) {
                            const cb = () => {
                                if (!this.isBuffering()) {
                                    defer();
                                }
                            };
                            const interval = setInterval(cb, 200);
                            yield promise;
                            clearInterval(interval);
                            logger_1.Logger.log('ActiveElement nudge (play)');
                            yield newElement.play();
                        }
                    }
                }
                else {
                    this.activeElement.syncClock(realTime);
                }
            }
            logger_1.Logger.groupEnd();
        });
    }
    seekTo(canvasId, time) {
        return __awaiter(this, void 0, void 0, function* () {
            this.updateActiveElement(canvasId, time);
            return this.seekToMediaTime(time);
        });
    }
    play(canvasId, time) {
        return __awaiter(this, void 0, void 0, function* () {
            this.playing = true;
            if (canvasId && typeof time !== 'undefined') {
                yield this.seekTo(canvasId, time);
            }
            if (this.activeElement) {
                logger_1.Logger.log(`CompositeMediaElement.play(${canvasId}, ${time})`);
                return this.activeElement.play(time).catch((err) => {
                    console.log('err', err);
                    this.playing = false;
                });
            }
        });
    }
    pause() {
        logger_1.Logger.log('Composite.pause()');
        this.playing = false;
        if (this.activeElement && !this.activeElement.isPaused()) {
            this.activeElement.pause();
        }
    }
    setVolume(volume) {
        for (const el of this.elements) {
            el.setVolume(volume);
        }
    }
    isBuffering() {
        return this.activeElement ? this.activeElement.isBuffering() : false;
    }
}
exports.CompositeMediaElement = CompositeMediaElement;
//# sourceMappingURL=composite-media-element.js.map