"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.TimePlanPlayer = void 0;
const relative_time_1 = require("../helpers/relative-time");
const logger_1 = require("../helpers/logger");
class TimePlanPlayer {
    constructor(media, plan, notifyRangeChange, notifyTimeChange, notifyPlaying, notifyBuffering) {
        this.continuous = true;
        this.playing = false;
        this._time = (0, relative_time_1.timelineTime)(0);
        logger_1.Logger.log('TimePlanPlayer', { media, plan });
        this.media = media;
        this.plan = plan;
        this.fullPlan = plan;
        this.currentStop = plan.stops[0];
        const noop = () => {
            // no-op.
        };
        this.notifyRangeChange = notifyRangeChange || noop;
        this.notifyTimeChange = notifyTimeChange || noop;
        this.notifyPlaying = notifyPlaying || noop;
        this.notifyBuffering = notifyBuffering || noop;
        this.logging = true;
        this.currentRange = this.currentStop.rangeStack[0];
        this.setTime(this.currentStop.start);
        media.onPlay((canvasId, time, el) => {
            // Playing the right thing...
            if (canvasId === this.currentStop.canvasId) {
                if (!this.playing) {
                    this.notifyPlaying(true);
                }
            }
            else {
                // el.pause();
            }
        });
        media.onBuffering(() => {
            this.notifyBuffering();
        });
        media.onPause((canvasId, time, el) => {
            if (el.isBuffering()) {
                return;
            }
            if (canvasId === this.currentStop.canvasId) {
                if (this.playing) {
                    this.notifyPlaying(false);
                }
            }
        });
    }
    selectPlan({ reset, rangeId } = {}) {
        if (reset) {
            return this.initialisePlan(this.fullPlan);
        }
        if (rangeId) {
            let foundStack = [];
            for (const plan of this.fullPlan.stops) {
                const idx = plan.rangeStack.indexOf(rangeId);
                if (plan.rangeStack.indexOf(rangeId) !== -1) {
                    foundStack = plan.rangeStack.slice(1, idx + 1);
                }
            }
            let plan = this.fullPlan;
            for (const id of foundStack) {
                for (const item of plan.items) {
                    if (item.type === 'time-plan' && item.rangeId === id) {
                        plan = item;
                        break;
                    }
                }
            }
            if (plan) {
                return this.initialisePlan(plan);
            }
        }
    }
    initialisePlan(plan) {
        this.plan = plan;
    }
    getCurrentRange() {
        const rangeId = this.currentRange;
        const isRangeWithStop = this.currentRange === this.currentStop.rangeId;
        const stopsToCheck = isRangeWithStop ? this.plan.stops : this.fullPlan.stops;
        const starting = [];
        const ending = [];
        for (const stop of stopsToCheck) {
            if (stop.rangeStack.indexOf(rangeId) !== -1) {
                starting.push(stop.start);
                ending.push(stop.end);
            }
            if (isRangeWithStop) {
                if (stop.rangeId === rangeId) {
                    starting.push(stop.start);
                    ending.push(stop.end);
                }
            }
        }
        const start = Math.min(...starting);
        const end = Math.max(...ending);
        logger_1.Logger.log('Range duration', {
            starting,
            ending,
            rangeId,
            isRangeWithStop,
            stopsToCheck,
            start: start - this.plan.start,
            end: end - this.plan.start,
            planStart: this.plan.start,
            duration: end - start,
            currentStop: this.currentStop,
        });
        return {
            start: start - this.plan.start,
            end: end - this.plan.start,
            duration: end - start,
        };
    }
    getTime() {
        return this._time;
    }
    setInternalTime(time) {
        this._time = time;
        this.notifyTimeChange(time);
        return this._time;
    }
    log(...content) {
        if (this.logging) {
            logger_1.Logger.log('TimePlanPlayer', ...content);
        }
    }
    setContinuousPlayback(continuous) {
        this.continuous = continuous;
    }
    setIsPlaying(playing) {
        this.playing = playing;
    }
    play() {
        this.log('Play', this.getTime());
        if (!this.playing) {
            this.setIsPlaying(true);
            this.media.play(this.currentStop.canvasId).catch((err) => {
                console.log('Err', err);
                this.setIsPlaying(false);
                this.notifyPlaying(false);
            });
        }
        return this.getTime();
    }
    currentTimelineTime() {
        return this.getTime();
    }
    currentMediaTime() {
        logger_1.Logger.log(`Current media time:
  - Current start: ${this.currentStop.start}
  - Current canvas: ${this.currentStop.canvasTime.start}
  - Current time: ${this.getTime()}
    `, this);
        const time = (0, relative_time_1.minusTime)(this.getTime(), this.currentStop.start);
        return (0, relative_time_1.annotationTime)((0, relative_time_1.addTime)(time, (0, relative_time_1.timelineTime)(this.currentStop.canvasTime.start)));
    }
    pause() {
        this.log('Pause', this.getTime());
        this.setIsPlaying(false);
        this.media.pause();
        return this.getTime();
    }
    setVolume(volume) {
        this.media.setVolume(volume);
    }
    findStop(time) {
        // // First check current stop.
        // if ((this.currentStop.start - 0.0001) <= time && (this.currentStop.end + 0.0001) > time) {
        //     return this.currentStop;
        // }
        //
        // // Then check next stop.
        // const idx = this.plan.stops.indexOf(this.currentStop);
        // const nextStop = idx !== -1 ? this.plan.stops[idx + 1] : undefined;
        // if (nextStop && nextStop.start <= time && nextStop.end > time) {
        //     return nextStop;
        // }
        // Fallback to checking all stops.
        for (const stop of this.plan.stops) {
            if (stop.start - 0.001 <= time && stop.end - 0.001 > time) {
                return stop;
            }
        }
        if (this.plan.stops[this.plan.stops.length - 1].end === time) {
            return this.plan.stops[this.plan.stops.length - 1];
        }
        return;
    }
    // Time that is set by the user.
    setTime(time, setRange = true) {
        return __awaiter(this, void 0, void 0, function* () {
            logger_1.Logger.groupCollapsed(`TimeplanPlayer.setTime(${time}, ${setRange ? 'true' : 'false'})`);
            // Early exit?
            const start = this.getTime();
            if (start !== time) {
                this.log('set time', { from: this.getTime(), to: time });
                this.setInternalTime(time);
                const stop = this.findStop(time);
                if (stop && stop !== this.currentStop) {
                    if (setRange) {
                        this.currentRange = stop.rangeId;
                    }
                    yield this.advanceToStop(this.currentStop, stop, undefined, time, setRange);
                }
            }
            logger_1.Logger.groupEnd();
        });
    }
    next() {
        return __awaiter(this, void 0, void 0, function* () {
            const currentRangeIndex = this.plan.rangeOrder.indexOf(this.currentRange);
            const isLast = currentRangeIndex >= 0 && currentRangeIndex === this.plan.rangeOrder.length - 1;
            const nextRangeIdx = !isLast ? this.plan.rangeOrder.indexOf(this.currentRange) + 1 : undefined;
            let nextRange = typeof nextRangeIdx !== 'undefined' ? this.plan.rangeOrder[nextRangeIdx] : undefined;
            const idx = this.plan.stops.indexOf(this.currentStop);
            let offset = 0;
            let nextStop = undefined;
            let running = true;
            while (running) {
                offset++;
                nextStop = this.plan.stops[idx + offset];
                if (!nextStop) {
                    running = false;
                    break;
                }
                if (!nextStop.noNav && nextStop.rangeId !== this.currentStop.rangeId) {
                    running = false;
                    break;
                }
            }
            if (this.playing && nextStop) {
                nextRange = nextStop.rangeId;
            }
            if (nextRange && nextStop && nextStop.rangeId !== nextRange) {
                if (this.playing ||
                    (this.currentStop.rangeStack.indexOf(nextRange) === -1 && nextStop.rangeStack.indexOf(nextRange) !== -1)) {
                    this.currentRange = this.playing ? nextStop.rangeId : nextRange;
                    this.setInternalTime(nextStop.start);
                    yield this.advanceToStop(this.currentStop, nextStop, this.playing ? nextStop.rangeId : nextRange);
                }
                else {
                    this.currentRange = nextRange;
                    this.setInternalTime(this.currentStop.start);
                    yield this.advanceToStop(this.currentStop, this.currentStop, nextRange);
                }
                return this.getTime();
            }
            if (nextStop) {
                this.setInternalTime(nextStop.start);
                this.currentRange = nextStop.rangeId;
                yield this.advanceToStop(this.currentStop, nextStop, nextStop.rangeId);
            }
            else {
                yield this.goToEndOfRange(this.currentStop.rangeId);
            }
            setTimeout(() => {
                if (this.playing) {
                    this.play();
                }
            }, 100);
            return this.getTime();
        });
    }
    goToEndOfRange(rangeId) {
        return __awaiter(this, void 0, void 0, function* () {
            let state = undefined;
            for (let i = 0; i < this.plan.stops.length; i++) {
                const stop = this.plan.stops[i];
                if (stop.rangeId === rangeId && (!state || (stop.canvasIndex >= state.canvasIndex && stop.end > state.end))) {
                    state = stop;
                }
            }
            if (state) {
                yield this.advanceToStop(this.currentStop, state, rangeId);
                this.setInternalTime(state.end);
            }
        });
    }
    goToStartOfRange(rangeId) {
        let state = undefined;
        const length = this.plan.stops.length;
        for (let i = length - 1; i >= 0; i--) {
            const stop = this.plan.stops[i];
            if (stop.rangeId === rangeId && (!state || (stop.canvasIndex <= state.canvasIndex && stop.start < state.start))) {
                state = stop;
            }
        }
        if (state) {
            if (state !== this.currentStop) {
                this.advanceToStop(this.currentStop, state, rangeId);
            }
            this.setInternalTime(state.start);
        }
    }
    previous() {
        const currentRangeIndex = this.plan.rangeOrder.indexOf(this.currentRange);
        const isFirst = currentRangeIndex === 0;
        const prevRangeIdx = !isFirst ? this.plan.rangeOrder.indexOf(this.currentRange) - 1 : undefined;
        let prevRange = typeof prevRangeIdx !== 'undefined' ? this.plan.rangeOrder[prevRangeIdx] : undefined;
        let currentStopHead = this.currentStop;
        const idx = this.plan.stops.indexOf(this.currentStop);
        let newIdx = idx;
        let prevStop = this.plan.stops[idx - 1];
        const firstStop = this.plan.stops[0];
        let running = true;
        const isValidNav = (plan) => {
            return !plan.noNav || plan === firstStop;
        };
        while (running) {
            const nextPrevStop = this.plan.stops[newIdx - 1];
            if (!nextPrevStop) {
                running = false;
                break;
            }
            if (isValidNav(nextPrevStop) && isValidNav(prevStop)) {
                if (nextPrevStop.rangeId === this.currentRange) {
                    currentStopHead = nextPrevStop;
                }
                if (prevStop.rangeId !== nextPrevStop.rangeId) {
                    running = false;
                    break;
                }
            }
            if (nextPrevStop) {
                if (isValidNav(nextPrevStop)) {
                    prevStop = nextPrevStop;
                }
                newIdx = newIdx - 1;
            }
        }
        const goBackToStartOfRange = this._time - (currentStopHead.start + 2) > 0;
        const isPreviousRangeDifferent = this.playing && prevStop && prevStop.rangeId !== this.currentStop.rangeId;
        const isDefinitelyFirstRange = idx === 0 || (!prevRange && newIdx === 0);
        const isPreviousRangeNotAParent = prevRange &&
            this.currentStop.rangeStack.indexOf(prevRange) === -1 &&
            // But it is in the previous.
            (prevStop.rangeStack.indexOf(prevRange) !== -1 || prevStop.rangeId === prevRange);
        const isPreviousRangeInStack = prevRange && this.currentStop.rangeStack.indexOf(prevRange) !== -1;
        logger_1.Logger.log('TimePlanPlayer.previous() => variables', {
            goBackToStartOfRange,
            isPreviousRangeDifferent,
            isDefinitelyFirstRange,
            isPreviousRangeNotAParent,
            isPreviousRangeInStack,
            prevStop,
        });
        if (goBackToStartOfRange) {
            logger_1.Logger.log('TimePlanPlayer.previous() => goBackToStartOfRange', { currentStopHead });
            if (currentStopHead !== this.currentStop) {
                this.advanceToStop(this.currentStop, currentStopHead, currentStopHead.rangeId);
            }
            this.setInternalTime(currentStopHead.start);
            return this.getTime();
        }
        if (isPreviousRangeDifferent) {
            prevRange = prevStop.rangeId;
        }
        // Case 1, at the start, but parent ranges possible.
        if (isDefinitelyFirstRange) {
            // Set the time to the start.
            this.goToStartOfRange(prevRange ? prevRange : this.currentStop.rangeId);
            // We are on the first item.
            if (prevRange && this.currentStop.rangeId !== prevRange) {
                // But we still want to change the range.
                this.currentRange = prevRange;
                this.advanceToStop(this.currentStop, currentStopHead, prevRange);
            }
            // And return the time.
            return this.getTime();
        }
        // Case 2, in the middle, but previous is a parent.
        if (prevRange && (isPreviousRangeNotAParent || prevStop === firstStop)) {
            // Then we navigate to the previous.
            this.setInternalTime(prevStop.start);
            this.currentRange = prevRange;
            this.advanceToStop(this.currentStop, prevStop, prevRange);
            // And time.
            return this.getTime();
        }
        // If the previous range is in the current ranges stack (i.e. a parent)
        if (prevRange && isPreviousRangeInStack) {
            this.setInternalTime(this.currentStop.start);
            this.currentRange = prevRange;
            this.advanceToStop(this.currentStop, currentStopHead, prevRange);
            // And time.
            return this.getTime();
        }
        return this.getTime();
    }
    setRange(id) {
        logger_1.Logger.log('setRange', id);
        if (id === this.currentRange) {
            return this.getTime();
        }
        this.currentRange = id;
        if (id === this.currentStop.rangeId) {
            // Or the start of the range?
            return this.getTime();
        }
        for (const stop of this.plan.stops) {
            if (stop.rangeId === id) {
                this.setInternalTime(stop.start);
                this.advanceToStop(this.currentStop, stop, id, undefined, !this.playing);
                break;
            }
        }
        for (const stop of this.plan.stops) {
            if (stop.rangeStack.indexOf(id) !== -1) {
                this.setInternalTime(stop.start);
                this.advanceToStop(this.currentStop, stop, id, undefined, !this.playing);
                break;
            }
        }
        return this.getTime();
    }
    isBuffering() {
        return this.media.isBuffering();
    }
    // Time that has ticked over.
    advanceToTime(time, paused) {
        logger_1.Logger.groupCollapsed(`TimeplanPlayer.advanceToTime(${time}, ${paused ? 'true' : 'false'})`);
        const stop = this.findStop(time);
        if (stop && this.currentStop !== stop) {
            logger_1.Logger.log('advanceToTime.a');
            this.setInternalTime(time);
            this.advanceToStop(this.currentStop, stop, undefined, time, paused);
            logger_1.Logger.groupEnd();
            return { buffering: this.isBuffering(), time };
        }
        // User has selected top level range.
        if (this.playing && this.currentRange !== this.currentStop.rangeId) {
            this.currentRange = this.currentStop.rangeId;
            this.notifyRangeChange(this.currentStop.rangeId, {
                from: this.currentStop,
                to: this.currentStop,
            });
        }
        if (!stop) {
            logger_1.Logger.log('advanceToTime.b');
            this.pause();
            this.setTime(this.currentStop.end);
            logger_1.Logger.groupEnd();
            return {
                paused: true,
                buffering: this.isBuffering(),
                time: this.currentStop.end,
            };
        }
        else {
            logger_1.Logger.log('advanceToTime.c', {
                time: this.getTime(),
            });
            this.setInternalTime(time);
            this.media.syncClock(this.currentMediaTime());
            logger_1.Logger.groupEnd();
            return { time };
        }
    }
    hasEnded() {
        return this.currentStop.end === this.getTime();
    }
    advanceToStop(from, to, rangeId, time, paused) {
        return __awaiter(this, void 0, void 0, function* () {
            logger_1.Logger.log('TimeplanPlayer.advanceToStop', {
                from,
                to,
                rangeId,
            });
            if (from === to) {
                if (rangeId) {
                    this.notifyRangeChange(rangeId ? rangeId : to.rangeId, {
                        to,
                        from,
                    });
                }
                return;
            }
            let promise;
            this.log('advanceToStop', to.start);
            const changeCanvas = this.currentStop.canvasId !== to.canvasId;
            this.currentStop = to;
            this.setInternalTime(typeof time !== 'undefined' ? time : to.start);
            if (changeCanvas && !paused) {
                promise = this.media.play(to.canvasId, this.currentMediaTime());
            }
            else {
                this.log(`advanceToStop -> seekToMediaTime.play(${this.currentMediaTime()})`);
                promise = this.media.seekToMediaTime(this.currentMediaTime(), to.canvasId);
            }
            this.notifyRangeChange(rangeId ? rangeId : to.rangeId, { to, from });
            yield promise;
        });
    }
    getStartTime() {
        return this.plan.start;
    }
    getDuration() {
        return this.plan.duration;
    }
}
exports.TimePlanPlayer = TimePlanPlayer;
//# sourceMappingURL=timeplan-player.js.map