"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const chalk_1 = require("chalk");
const build_1 = require("../lib/build");
const command_1 = require("../lib/command");
const errors_1 = require("../lib/errors");
class BuildCommand extends command_1.Command {
    getMetadata() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const options = [];
            const footnotes = [];
            const exampleCommands = [''];
            let description = `${chalk_1.default.green('ionic build')} will perform an Ionic build, which compiles web assets and prepares them for deployment.`;
            let groups = [];
            const runner = this.project && (yield this.project.getBuildRunner());
            if (runner) {
                const libmetadata = yield runner.getCommandMetadata();
                groups = libmetadata.groups || [];
                options.push(...libmetadata.options || []);
                description += libmetadata.description ? `\n\n${libmetadata.description.trim()}` : '';
                footnotes.push(...libmetadata.footnotes || []);
                exampleCommands.push(...libmetadata.exampleCommands || []);
            }
            options.push(...build_1.COMMON_BUILD_COMMAND_OPTIONS);
            return {
                name: 'build',
                type: 'project',
                summary: 'Build web assets and prepare your app for any platform targets',
                description,
                footnotes,
                groups,
                exampleCommands,
                options,
            };
        });
    }
    preRun(inputs, options) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            if (inputs.length > 0 && ['android', 'ios', 'wp8', 'windows', 'browser'].includes(inputs[0])) {
                this.env.log.warn(`${chalk_1.default.green('ionic build')} is for building web assets and takes no arguments. See ${chalk_1.default.green('ionic build --help')}.\n` +
                    `Ignoring argument ${chalk_1.default.green(inputs[0])}. Perhaps you meant ${chalk_1.default.green('ionic cordova build ' + inputs[0])}?\n`);
                inputs.splice(0);
            }
        });
    }
    run(inputs, options, runinfo) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            if (!this.project) {
                throw new errors_1.FatalException(`Cannot run ${chalk_1.default.green('ionic build')} outside a project directory.`);
            }
            // TODO: use runner directly
            yield build_1.build({ config: this.env.config, log: this.env.log, shell: this.env.shell, prompt: this.env.prompt, project: this.project }, inputs, options);
        });
    }
}
exports.BuildCommand = BuildCommand;
