"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const cli_framework_1 = require("@ionic/cli-framework");
const utils_fs_1 = require("@ionic/utils-fs");
const chalk_1 = require("chalk");
const path = require("path");
const command_1 = require("../../lib/command");
const errors_1 = require("../../lib/errors");
const executor_1 = require("../../lib/executor");
class CapacitorCommand extends command_1.Command {
    get integration() {
        if (!this.project) {
            throw new errors_1.FatalException(`Cannot use Capacitor outside a project directory.`);
        }
        if (!this._integration) {
            this._integration = this.project.requireIntegration('capacitor');
        }
        return this._integration;
    }
    checkCapacitor(runinfo) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            if (!this.project) {
                throw new errors_1.FatalException(`Cannot use Capacitor outside a project directory.`);
            }
            const capacitor = this.project.getIntegration('capacitor');
            if (!capacitor) {
                yield executor_1.runCommand(runinfo, ['integrations', 'enable', 'capacitor']);
            }
        });
    }
    preRunChecks(runinfo) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            yield this.checkCapacitor(runinfo);
        });
    }
    runCapacitor(argList) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            try {
                return yield this._runCapacitor(argList);
            }
            catch (e) {
                if (e instanceof cli_framework_1.ShellCommandError) {
                    if (e.code === cli_framework_1.ERROR_SHELL_COMMAND_NOT_FOUND) {
                        const pkg = '@capacitor/cli';
                        const requiredMsg = `The Capacitor CLI is required for Capacitor projects.`;
                        this.env.log.nl();
                        this.env.log.info(`Looks like ${chalk_1.default.green(pkg)} isn't installed in this project.\n` + requiredMsg);
                        this.env.log.nl();
                        const installed = yield this.promptToInstallCapacitor();
                        if (!installed) {
                            throw new errors_1.FatalException(`${chalk_1.default.green(pkg)} is required for Capacitor projects.`);
                        }
                        return this.runCapacitor(argList);
                    }
                    if (e.code === cli_framework_1.ERROR_SHELL_SIGNAL_EXIT) {
                        return;
                    }
                }
                throw e;
            }
        });
    }
    checkForPlatformInstallation(platform) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            if (!this.project) {
                throw new errors_1.FatalException('Cannot use Capacitor outside a project directory.');
            }
            if (platform) {
                const integrationRoot = this.project.directory;
                const platformsToCheck = ['android', 'ios', 'electron'];
                const platforms = (yield Promise.all(platformsToCheck.map((p) => tslib_1.__awaiter(this, void 0, void 0, function* () { return [p, yield utils_fs_1.pathExists(path.resolve(integrationRoot, p))]; }))))
                    .filter(([, e]) => e)
                    .map(([p]) => p);
                if (!platforms.includes(platform)) {
                    yield this._runCapacitor(['add', platform]);
                }
            }
        });
    }
    promptToInstallCapacitor() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            if (!this.project) {
                throw new errors_1.FatalException(`Cannot use Capacitor outside a project directory.`);
            }
            const { pkgManagerArgs } = yield Promise.resolve().then(() => require('../../lib/utils/npm'));
            const pkg = '@capacitor/cli';
            const [manager, ...managerArgs] = yield pkgManagerArgs(this.env.config.get('npmClient'), { pkg, command: 'install', saveDev: true });
            const confirm = yield this.env.prompt({
                name: 'confirm',
                message: `Install ${chalk_1.default.green(pkg)}?`,
                type: 'confirm',
            });
            if (!confirm) {
                this.env.log.warn(`Not installing--here's how to install manually: ${chalk_1.default.green(`${manager} ${managerArgs.join(' ')}`)}`);
                return false;
            }
            yield this.env.shell.run(manager, managerArgs, { cwd: this.project.directory });
            return true;
        });
    }
    _runCapacitor(argList) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            if (!this.project) {
                throw new errors_1.FatalException(`Cannot use Capacitor outside a project directory.`);
            }
            yield this.env.shell.run('capacitor', argList, { fatalOnNotFound: false, truncateErrorOutput: 5000, stdio: 'inherit', cwd: this.integration.root });
        });
    }
}
exports.CapacitorCommand = CapacitorCommand;
