"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const cli_framework_1 = require("@ionic/cli-framework");
const process_1 = require("@ionic/cli-framework/utils/process");
const chalk_1 = require("chalk");
const path = require("path");
const build_1 = require("../../lib/build");
const errors_1 = require("../../lib/errors");
const config_1 = require("../../lib/integrations/capacitor/config");
const utils_1 = require("../../lib/integrations/capacitor/utils");
const serve_1 = require("../../lib/serve");
const base_1 = require("./base");
class RunCommand extends base_1.CapacitorCommand {
    getMetadata() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            let groups = [cli_framework_1.CommandGroup.Beta];
            const exampleCommands = [
                '',
                'android',
                'android -l',
                'ios --livereload',
                'ios --livereload-url=http://localhost:8100',
            ].sort();
            const options = [
                // Build Options
                {
                    name: 'build',
                    summary: 'Do not invoke Ionic build',
                    type: Boolean,
                    default: true,
                },
                ...serve_1.COMMON_SERVE_COMMAND_OPTIONS.filter(o => !['livereload'].includes(o.name)),
                {
                    name: 'livereload',
                    summary: 'Spin up dev server to live-reload www files',
                    type: Boolean,
                    aliases: ['l'],
                },
                {
                    name: 'livereload-url',
                    summary: 'Provide a custom URL to the dev server',
                    spec: { value: 'url' },
                },
            ];
            const footnotes = [
                {
                    id: 'remote-debugging-docs',
                    url: 'https://ionicframework.com/docs/developer-resources/developer-tips',
                    shortUrl: 'https://ion.link/remote-debugging-docs',
                },
            ];
            const serveRunner = this.project && (yield this.project.getServeRunner());
            const buildRunner = this.project && (yield this.project.getBuildRunner());
            if (buildRunner) {
                const libmetadata = yield buildRunner.getCommandMetadata();
                groups = libmetadata.groups || [];
                options.push(...libmetadata.options || []);
                footnotes.push(...libmetadata.footnotes || []);
            }
            if (serveRunner) {
                const libmetadata = yield serveRunner.getCommandMetadata();
                const existingOpts = options.map(o => o.name);
                groups = libmetadata.groups || [];
                options.push(...(libmetadata.options || []).filter(o => !existingOpts.includes(o.name)).map(o => (Object.assign({}, o, { hint: `${o.hint ? `${o.hint} ` : ''}${chalk_1.default.dim('(--livereload)')}` }))));
                footnotes.push(...libmetadata.footnotes || []);
            }
            return {
                name: 'run',
                type: 'project',
                summary: 'Run an Ionic project on a connected device',
                description: `
${chalk_1.default.green('ionic capacitor run')} will do the following:
- Perform ${chalk_1.default.green('ionic build')} (or run the dev server from ${chalk_1.default.green('ionic serve')} with the ${chalk_1.default.green('--livereload')} option)
- Copy web assets into the specified native platform
- Open the IDE for your native project (Xcode for iOS, Android Studio for Android)

Once the web assets and configuration are copied into your native project, the app can run on devices and emulators/simulators using the native IDE. Unfortunately, programmatically building and launching the native project is not yet supported.

For Android and iOS, you can setup Remote Debugging on your device with browser development tools using these docs[^remote-debugging-docs].
      `,
                footnotes,
                exampleCommands,
                inputs: [
                    {
                        name: 'platform',
                        summary: `The platform to run (e.g. ${['android', 'ios'].map(v => chalk_1.default.green(v)).join(', ')})`,
                        validators: [cli_framework_1.validators.required],
                    },
                ],
                options,
                groups,
            };
        });
    }
    preRun(inputs, options, runinfo) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            yield this.preRunChecks(runinfo);
            if (!inputs[0]) {
                const platform = yield this.env.prompt({
                    type: 'list',
                    name: 'platform',
                    message: 'What platform would you like to run?',
                    choices: ['android', 'ios'],
                });
                inputs[0] = platform.trim();
            }
            if (options['livereload-url']) {
                options['livereload'] = true;
            }
            if (!options['build'] && options['livereload']) {
                this.env.log.warn(`No livereload with ${chalk_1.default.green('--no-build')}.`);
                options['livereload'] = false;
            }
            yield this.checkForPlatformInstallation(inputs[0]);
        });
    }
    run(inputs, options) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            if (!this.project) {
                throw new errors_1.FatalException(`Cannot run ${chalk_1.default.green('ionic cordova run/emulate')} outside a project directory.`);
            }
            const [platform] = inputs;
            if (options['livereload']) {
                let livereloadUrl = options['livereload-url'] ? String(options['livereload-url']) : undefined;
                if (!livereloadUrl) {
                    // TODO: use runner directly
                    const details = yield serve_1.serve({ flags: this.env.flags, config: this.env.config, log: this.env.log, prompt: this.env.prompt, shell: this.env.shell, project: this.project }, inputs, utils_1.generateOptionsForCapacitorBuild(inputs, options));
                    if (details.externallyAccessible === false) {
                        const extra = serve_1.LOCAL_ADDRESSES.includes(details.externalAddress) ? '\nEnsure you have proper port forwarding setup from your device to your computer.' : '';
                        this.env.log.warn(`Your device or emulator may not be able to access ${chalk_1.default.bold(details.externalAddress)}.${extra}\n\n`);
                    }
                    livereloadUrl = `${details.protocol || 'http'}://${details.externalAddress}:${details.port}`;
                }
                const conf = new config_1.CapacitorConfig(path.resolve(this.project.directory, config_1.CAPACITOR_CONFIG_FILE));
                process_1.onBeforeExit(() => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    conf.resetServerUrl();
                }));
                conf.setServerUrl(livereloadUrl);
            }
            else {
                if (options['build']) {
                    // TODO: use runner directly
                    yield build_1.build({ config: this.env.config, log: this.env.log, shell: this.env.shell, prompt: this.env.prompt, project: this.project }, inputs, utils_1.generateOptionsForCapacitorBuild(inputs, options));
                }
            }
            // copy assets and capacitor.config.json into place
            yield this.runCapacitor(['copy', platform]);
            // TODO: native-run
            this.env.log.nl();
            this.env.log.info('Ready for use in your Native IDE!\n' +
                `To continue, run your project on a device or ${platform === 'ios' ? 'simulator' : 'emulator'} using ${platform === 'ios' ? 'Xcode' : 'Android Studio'}!`);
            this.env.log.nl();
            yield this.runCapacitor(['open', platform]);
            if (options['livereload']) {
                this.env.log.nl();
                this.env.log.info('Development server will continue running until manually stopped.\n' +
                    chalk_1.default.yellow('Use Ctrl+C to quit this process'));
                yield process_1.sleepForever();
            }
        });
    }
}
exports.RunCommand = RunCommand;
