"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const cli_framework_1 = require("@ionic/cli-framework");
const format_1 = require("@ionic/cli-framework/utils/format");
const string_1 = require("@ionic/cli-framework/utils/string");
const chalk_1 = require("chalk");
const lodash = require("lodash");
const util = require("util");
const constants_1 = require("../../constants");
const base_1 = require("./base");
class ConfigGetCommand extends base_1.BaseConfigCommand {
    getMetadata() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const projectFile = this.project ? format_1.prettyPath(this.project.filePath) : constants_1.PROJECT_FILE;
            return {
                name: 'get',
                type: 'global',
                summary: 'Print config values',
                description: `
This command reads and prints configuration values from the project's ${chalk_1.default.bold(projectFile)} file. It can also operate on the global CLI configuration (${chalk_1.default.bold('~/.ionic/config.json')}) using the ${chalk_1.default.green('--global')} option.

For nested properties, separate nest levels with dots. For example, the property name ${chalk_1.default.green('integrations.cordova')} will look in the ${chalk_1.default.bold('integrations')} object for the ${chalk_1.default.bold('cordova')} property.

Without a ${chalk_1.default.green('property')} argument, this command prints out the entire config.

For multi-app projects, this command is scoped to the current project by default. To operate at the root of the project configuration file instead, use the ${chalk_1.default.green('--root')} option.

If you are using this command programmatically, you can use the ${chalk_1.default.green('--json')} option.

This command will sanitize config output for known sensitive fields (disabled when using ${chalk_1.default.green('--json')}).
      `,
                inputs: [
                    {
                        name: 'property',
                        summary: 'The property name you wish to get',
                    },
                ],
                options: [
                    {
                        name: 'global',
                        summary: 'Use global CLI config',
                        type: Boolean,
                        aliases: ['g'],
                    },
                    {
                        name: 'json',
                        summary: 'Output config values in JSON',
                        type: Boolean,
                        groups: [cli_framework_1.OptionGroup.Advanced],
                    },
                    {
                        name: 'root',
                        summary: `Operate on root of ${chalk_1.default.bold(projectFile)}`,
                        type: Boolean,
                        hint: chalk_1.default.dim('[multi-app]'),
                        groups: [cli_framework_1.OptionGroup.Advanced],
                    },
                ],
                exampleCommands: ['', 'id', '--global user.email', '-g npmClient'],
            };
        });
    }
    run(inputs, options) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const ctx = this.generateContext(inputs, options);
            const conf = base_1.getConfigValue(ctx);
            this.printConfig(ctx, conf);
        });
    }
    printConfig(ctx, v) {
        const { global, json } = ctx;
        if (json) {
            process.stdout.write(this.jsonStringify(v));
        }
        else {
            if (global && v && typeof v === 'object') {
                const columns = lodash.entries(v)
                    .map(([k, v]) => [k, this.sanitizeEntry(k, v)])
                    .map(([k, v]) => [chalk_1.default.bold(k), util.inspect(v, { colors: chalk_1.default.enabled })]);
                columns.sort((a, b) => string_1.strcmp(a[0], b[0]));
                this.env.log.rawmsg(format_1.columnar(columns, {}));
            }
            else {
                this.env.log.rawmsg(util.inspect(v, { depth: Infinity, colors: chalk_1.default.enabled }));
            }
        }
    }
    sanitizeEntry(key, value) {
        if (key.includes('tokens')) {
            return '*****';
        }
        return value;
    }
}
exports.ConfigGetCommand = ConfigGetCommand;
