"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const cli_framework_1 = require("@ionic/cli-framework");
const format_1 = require("@ionic/cli-framework/utils/format");
const chalk_1 = require("chalk");
const constants_1 = require("../../constants");
const errors_1 = require("../../lib/errors");
const base_1 = require("./base");
class ConfigSetCommand extends base_1.BaseConfigCommand {
    getMetadata() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const projectFile = this.project ? format_1.prettyPath(this.project.filePath) : constants_1.PROJECT_FILE;
            return {
                name: 'set',
                type: 'global',
                summary: 'Set config values',
                description: `
This command writes configuration values to the project's ${chalk_1.default.bold(format_1.prettyPath(projectFile))} file. It can also operate on the global CLI configuration (${chalk_1.default.bold('~/.ionic/config.json')}) using the ${chalk_1.default.green('--global')} option.

For nested properties, separate nest levels with dots. For example, the property name ${chalk_1.default.green('integrations.cordova')} will look in the ${chalk_1.default.bold('integrations')} object for the ${chalk_1.default.bold('cordova')} property.

For multi-app projects, this command is scoped to the current project by default. To operate at the root of the project configuration file instead, use the ${chalk_1.default.green('--root')} option.

This command will attempt to coerce ${chalk_1.default.green('value')} into a suitable JSON type. If it is JSON-parsable, such as ${chalk_1.default.green('123')}, ${chalk_1.default.green('true')}, ${chalk_1.default.green('[]')}, etc., then it takes the parsed result. Otherwise, the value is interpreted as a string. For stricter input, use ${chalk_1.default.green('--json')}, which will error with non-JSON values.

By default, if ${chalk_1.default.green('property')} exists and is an object or an array, the value is not overwritten. To disable this check and always overwrite the property, use ${chalk_1.default.green('--force')}.
      `,
                inputs: [
                    {
                        name: 'property',
                        summary: 'The property name you wish to set',
                        validators: [cli_framework_1.validators.required],
                    },
                    {
                        name: 'value',
                        summary: 'The new value of the given property',
                        validators: [cli_framework_1.validators.required],
                    },
                ],
                options: [
                    {
                        name: 'global',
                        summary: 'Use global CLI config',
                        type: Boolean,
                        aliases: ['g'],
                    },
                    {
                        name: 'json',
                        summary: `Always interpret ${chalk_1.default.green('value')} as JSON`,
                        type: Boolean,
                        groups: [cli_framework_1.OptionGroup.Advanced],
                    },
                    {
                        name: 'force',
                        summary: 'Always overwrite existing values',
                        type: Boolean,
                        groups: [cli_framework_1.OptionGroup.Advanced],
                    },
                    {
                        name: 'root',
                        summary: `Operate on root of ${chalk_1.default.bold(format_1.prettyPath(projectFile))}`,
                        type: Boolean,
                        hint: chalk_1.default.dim('[multi-app]'),
                        groups: [cli_framework_1.OptionGroup.Advanced],
                    },
                ],
                exampleCommands: ['name newAppName', 'name "\\"newAppName\\"" --json', '-g interactive false'],
            };
        });
    }
    run(inputs, options) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const ctx = this.generateContext(inputs, options);
            const { property } = ctx;
            if (typeof property === 'undefined') {
                throw new errors_1.FatalException(`Cannot set config to ${chalk_1.default.green(ctx.value)} without a property.`);
            }
            const originalValue = base_1.getConfigValue(ctx);
            base_1.setConfigValue(Object.assign({}, ctx, { property, originalValue }));
            if (ctx.value !== originalValue) {
                this.env.log.ok(`${chalk_1.default.green(property)} set to ${chalk_1.default.green(JSON.stringify(ctx.value))}!`);
            }
            else {
                this.env.log.info(`${chalk_1.default.green(property)} is already set to ${chalk_1.default.green(JSON.stringify(ctx.value))}.`);
            }
        });
    }
}
exports.ConfigSetCommand = ConfigSetCommand;
