"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const cli_framework_1 = require("@ionic/cli-framework");
const format_1 = require("@ionic/cli-framework/utils/format");
const chalk_1 = require("chalk");
const constants_1 = require("../../constants");
const errors_1 = require("../../lib/errors");
const base_1 = require("./base");
class ConfigUnsetCommand extends base_1.BaseConfigCommand {
    getMetadata() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const projectFile = this.project ? format_1.prettyPath(this.project.filePath) : constants_1.PROJECT_FILE;
            return {
                name: 'unset',
                type: 'global',
                summary: 'Delete config values',
                description: `
This command deletes configuration values from the project's ${chalk_1.default.bold(format_1.prettyPath(projectFile))} file. It can also operate on the global CLI configuration (${chalk_1.default.bold('~/.ionic/config.json')}) using the ${chalk_1.default.green('--global')} option.

For nested properties, separate nest levels with dots. For example, the property name ${chalk_1.default.green('integrations.cordova')} will look in the ${chalk_1.default.bold('integrations')} object for the ${chalk_1.default.bold('cordova')} property.

For multi-app projects, this command is scoped to the current project by default. To operate at the root of the project configuration file instead, use the ${chalk_1.default.green('--root')} option.
      `,
                inputs: [
                    {
                        name: 'property',
                        summary: 'The property name you wish to delete',
                        validators: [cli_framework_1.validators.required],
                    },
                ],
                options: [
                    {
                        name: 'global',
                        summary: 'Use global CLI config',
                        type: Boolean,
                        aliases: ['g'],
                    },
                    {
                        name: 'root',
                        summary: `Operate on root of ${chalk_1.default.bold(format_1.prettyPath(projectFile))}`,
                        type: Boolean,
                        hint: chalk_1.default.dim('[multi-app]'),
                        groups: [cli_framework_1.OptionGroup.Advanced],
                    },
                ],
                exampleCommands: ['', 'type', '--global git.setup', '-g interactive'],
            };
        });
    }
    run(inputs, options) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const ctx = this.generateContext(inputs, options);
            const { property } = ctx;
            if (typeof property === 'undefined') {
                throw new errors_1.FatalException(`Cannot unset config entry without a property.`);
            }
            const propertyExists = typeof base_1.getConfigValue(ctx) !== 'undefined';
            base_1.unsetConfigValue(Object.assign({}, ctx, { property }));
            if (propertyExists) {
                this.env.log.ok(`${chalk_1.default.green(property)} unset!`);
            }
            else {
                this.env.log.warn(`Property ${chalk_1.default.green(property)} does not exist--cannot unset.`);
            }
        });
    }
}
exports.ConfigUnsetCommand = ConfigUnsetCommand;
