"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const cli_framework_1 = require("@ionic/cli-framework");
const format_1 = require("@ionic/cli-framework/utils/format");
const utils_fs_1 = require("@ionic/utils-fs");
const chalk_1 = require("chalk");
const path = require("path");
const command_1 = require("../../lib/command");
const errors_1 = require("../../lib/errors");
const executor_1 = require("../../lib/executor");
exports.CORDOVA_COMPILE_OPTIONS = [
    {
        name: 'debug',
        summary: 'Mark as a debug build',
        type: Boolean,
        groups: ['cordova', 'cordova-cli'],
        hint: chalk_1.default.dim('[cordova]'),
    },
    {
        name: 'release',
        summary: 'Mark as a release build',
        type: Boolean,
        groups: ['cordova', 'cordova-cli'],
        hint: chalk_1.default.dim('[cordova]'),
    },
    {
        name: 'device',
        summary: 'Deploy build to a device',
        type: Boolean,
        groups: ['cordova', 'cordova-cli', 'native-run'],
        hint: chalk_1.default.dim('[cordova]'),
    },
    {
        name: 'emulator',
        summary: 'Deploy build to an emulator',
        type: Boolean,
        groups: ['cordova', 'cordova-cli', 'native-run'],
        hint: chalk_1.default.dim('[cordova]'),
    },
    {
        name: 'buildConfig',
        summary: 'Use the specified build configuration',
        groups: [cli_framework_1.OptionGroup.Advanced, 'cordova', 'cordova-cli'],
        hint: chalk_1.default.dim('[cordova]'),
        spec: { value: 'file' },
    },
];
exports.CORDOVA_RUN_OPTIONS = [
    ...exports.CORDOVA_COMPILE_OPTIONS,
    {
        name: 'target',
        summary: `Deploy build to a device (use ${chalk_1.default.green('--list')} to see all)`,
        type: String,
        groups: [cli_framework_1.OptionGroup.Advanced, 'cordova', 'cordova-cli', 'native-run'],
        hint: chalk_1.default.dim('[cordova]'),
    },
];
exports.CORDOVA_BUILD_EXAMPLE_COMMANDS = [
    'ios',
    'ios --prod --release',
    'ios --prod --release -- --developmentTeam="ABCD" --codeSignIdentity="iPhone Developer" --packageType="app-store"',
    'ios --buildConfig=build.json',
    'ios --prod --release --buildConfig=build.json',
    'android',
    'android --prod --release -- -- --keystore=filename.keystore --alias=myalias',
    'android --prod --release -- -- --minSdkVersion=21',
    'android --prod --release -- -- --versionCode=55',
    'android --prod --release -- -- --gradleArg=-PcdvBuildMultipleApks=true',
    'android --buildConfig=build.json',
    'android --prod --release --buildConfig=build.json',
];
class CordovaCommand extends command_1.Command {
    get integration() {
        if (!this.project) {
            throw new errors_1.FatalException(`Cannot use Cordova outside a project directory.`);
        }
        if (!this._integration) {
            this._integration = this.project.requireIntegration('cordova');
        }
        return this._integration;
    }
    checkCordova(runinfo) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            if (!this.project) {
                throw new errors_1.FatalException('Cannot use Cordova outside a project directory.');
            }
            const cordova = this.project.getIntegration('cordova');
            if (!cordova) {
                yield executor_1.runCommand(runinfo, ['integrations', 'enable', 'cordova']);
            }
        });
    }
    preRunChecks(runinfo) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            if (!this.project) {
                throw new errors_1.FatalException('Cannot use Cordova outside a project directory.');
            }
            const { loadConfigXml } = yield Promise.resolve().then(() => require('../../lib/integrations/cordova/config'));
            yield this.checkCordova(runinfo);
            // Check for www folder
            if (this.project.directory) {
                const wwwPath = path.join(this.integration.root, 'www');
                const wwwExists = yield utils_fs_1.pathExists(wwwPath); // TODO: hard-coded
                if (!wwwExists) {
                    const tasks = this.createTaskChain();
                    tasks.next(`Creating ${chalk_1.default.bold(format_1.prettyPath(wwwPath))} directory for you`);
                    yield utils_fs_1.mkdirp(wwwPath);
                    tasks.end();
                }
            }
            const conf = yield loadConfigXml(this.integration);
            conf.resetContentSrc();
            yield conf.save();
        });
    }
    runCordova(argList, _a = {}) {
        var { fatalOnNotFound = false, truncateErrorOutput = 5000 } = _a, options = tslib_1.__rest(_a, ["fatalOnNotFound", "truncateErrorOutput"]);
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            if (!this.project) {
                throw new errors_1.FatalException('Cannot use Cordova outside a project directory.');
            }
            const { pkgManagerArgs } = yield Promise.resolve().then(() => require('../../lib/utils/npm'));
            try {
                yield this.env.shell.run('cordova', argList, Object.assign({ fatalOnNotFound, truncateErrorOutput, cwd: this.integration.root }, options));
            }
            catch (e) {
                if (e instanceof cli_framework_1.ShellCommandError) {
                    if (e.code === cli_framework_1.ERROR_SHELL_COMMAND_NOT_FOUND) {
                        const cdvInstallArgs = yield pkgManagerArgs(this.env.config.get('npmClient'), { command: 'install', pkg: 'cordova', global: true });
                        throw new errors_1.FatalException(`The Cordova CLI was not found on your PATH. Please install Cordova globally:\n` +
                            `${chalk_1.default.green(cdvInstallArgs.join(' '))}\n`);
                    }
                    if (e.code === cli_framework_1.ERROR_SHELL_SIGNAL_EXIT) {
                        return;
                    }
                }
                if (options.fatalOnError) {
                    this.env.log.nl();
                    this.env.log.error('Cordova encountered an error.\nYou may get more insight by running the Cordova command above directly.\n');
                }
                throw e;
            }
        });
    }
    checkForPlatformInstallation(platform, { promptToInstall = false, promptToInstallRefusalMsg = `Cannot run this command for the ${chalk_1.default.green(platform)} platform unless it is installed.` } = {}) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            if (!this.project) {
                throw new errors_1.FatalException('Cannot use Cordova outside a project directory.');
            }
            if (platform) {
                const { getPlatforms } = yield Promise.resolve().then(() => require('../../lib/integrations/cordova/project'));
                const platforms = yield getPlatforms(this.integration.root);
                if (!platforms.includes(platform)) {
                    const confirm = promptToInstall ? yield this.env.prompt({
                        message: `Platform ${chalk_1.default.green(platform)} is not installed! Would you like to install it?`,
                        type: 'confirm',
                        name: 'confirm',
                    }) : true;
                    if (confirm) {
                        yield this.runCordova(['platform', 'add', platform, '--save']);
                    }
                    else {
                        throw new errors_1.FatalException(promptToInstallRefusalMsg);
                    }
                }
            }
        });
    }
}
exports.CordovaCommand = CordovaCommand;
