"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const cli_framework_1 = require("@ionic/cli-framework");
const chalk_1 = require("chalk");
const build_1 = require("../../lib/build");
const errors_1 = require("../../lib/errors");
const utils_1 = require("../../lib/integrations/cordova/utils");
const base_1 = require("./base");
class BuildCommand extends base_1.CordovaCommand {
    getMetadata() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const exampleCommands = base_1.CORDOVA_BUILD_EXAMPLE_COMMANDS.sort();
            const options = [
                // Build Options
                {
                    name: 'build',
                    summary: 'Do not invoke an Ionic build',
                    type: Boolean,
                    default: true,
                },
                ...base_1.CORDOVA_COMPILE_OPTIONS,
            ];
            const footnotes = [
                {
                    id: 'cordova-android-using-flags',
                    url: 'https://cordova.apache.org/docs/en/latest/guide/platforms/android/index.html#using-flags',
                },
                {
                    id: 'cordova-ios-using-flags',
                    url: 'https://cordova.apache.org/docs/en/latest/guide/platforms/ios/index.html#using-flags',
                },
            ];
            const runner = this.project && (yield this.project.getBuildRunner());
            if (runner) {
                const libmetadata = yield runner.getCommandMetadata();
                options.push(...libmetadata.options || []);
                footnotes.push(...libmetadata.footnotes || []);
            }
            return {
                name: 'build',
                type: 'project',
                summary: 'Build (prepare + compile) an Ionic project for a given platform',
                description: `
Like running ${chalk_1.default.green('cordova build')} directly, but also builds web assets with configuration from ${chalk_1.default.green('ionic build')} and provides friendly checks.

To pass additional options to the Cordova CLI, use the ${chalk_1.default.green('--')} separator after the Ionic CLI arguments.

The Cordova CLI requires a separator for platform-specific arguments for Android builds[^cordova-android-using-flags], so an additional separator is required for the Ionic CLI, but it is not required for iOS builds[^cordova-ios-using-flags]. See the example commands for usage with separators. To avoid using flags, consider using ${chalk_1.default.green('--buildConfig')} with a ${chalk_1.default.bold('build.json')} file.
      `,
                footnotes,
                exampleCommands,
                inputs: [
                    {
                        name: 'platform',
                        summary: `The platform to build (e.g. ${['android', 'ios'].map(v => chalk_1.default.green(v)).join(', ')})`,
                        validators: [cli_framework_1.validators.required],
                    },
                ],
                options,
            };
        });
    }
    preRun(inputs, options, runinfo) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            yield this.preRunChecks(runinfo);
            if (!inputs[0]) {
                const platform = yield this.env.prompt({
                    type: 'input',
                    name: 'platform',
                    message: `What platform would you like to build (${['android', 'ios'].map(v => chalk_1.default.green(v)).join(', ')}):`,
                });
                inputs[0] = platform.trim();
            }
            yield this.checkForPlatformInstallation(inputs[0]);
        });
    }
    run(inputs, options) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const metadata = yield this.getMetadata();
            if (!this.project) {
                throw new errors_1.FatalException(`Cannot run ${chalk_1.default.green('ionic cordova build')} outside a project directory.`);
            }
            if (options.build) {
                // TODO: use runner directly
                yield build_1.build({ config: this.env.config, log: this.env.log, shell: this.env.shell, prompt: this.env.prompt, project: this.project }, inputs, utils_1.generateOptionsForCordovaBuild(metadata, inputs, options));
            }
            const cordovaArgs = utils_1.filterArgumentsForCordova(metadata, options);
            yield this.runCordova(cordovaArgs, {});
        });
    }
}
exports.BuildCommand = BuildCommand;
