"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const cli_framework_1 = require("@ionic/cli-framework");
const chalk_1 = require("chalk");
const lodash = require("lodash");
const errors_1 = require("../../lib/errors");
const executor_1 = require("../../lib/executor");
const base_1 = require("./base");
class PlatformCommand extends base_1.CordovaCommand {
    getMetadata() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return {
                name: 'platform',
                type: 'project',
                summary: 'Manage Cordova platform targets',
                description: `
Like running ${chalk_1.default.green('cordova platform')} directly, but adds default Ionic icons and splash screen resources (during ${chalk_1.default.green('add')}) and provides friendly checks.
      `,
                exampleCommands: ['', 'add ios', 'add android', 'rm ios'],
                inputs: [
                    {
                        name: 'action',
                        summary: `${chalk_1.default.green('add')}, ${chalk_1.default.green('remove')}, or ${chalk_1.default.green('update')} a platform; ${chalk_1.default.green('ls')}, ${chalk_1.default.green('check')}, or ${chalk_1.default.green('save')} all project platforms`,
                    },
                    {
                        name: 'platform',
                        summary: `The platform that you would like to add (${['android', 'ios'].map(v => chalk_1.default.green(v)).join(', ')})`,
                    },
                ],
                options: [
                    {
                        name: 'resources',
                        summary: `Do not pregenerate icons and splash screen resources (corresponds to ${chalk_1.default.green('add')})`,
                        type: Boolean,
                        default: true,
                    },
                ],
            };
        });
    }
    preRun(inputs, options, runinfo) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            yield this.preRunChecks(runinfo);
            if (options['r'] || options['noresources']) {
                options['resources'] = false;
            }
            inputs[0] = !inputs[0] ? 'ls' : inputs[0];
            inputs[0] = (inputs[0] === 'rm') ? 'remove' : inputs[0];
            inputs[0] = (inputs[0] === 'list') ? 'ls' : inputs[0];
            cli_framework_1.validate(inputs[0], 'action', [cli_framework_1.contains(['add', 'remove', 'update', 'ls', 'check', 'save'], {})]);
            // If the action is list, check, or save, then just end here.
            if (['ls', 'check', 'save'].includes(inputs[0])) {
                yield this.runCordova(['platform', inputs[0]], {});
                throw new errors_1.FatalException('', 0);
            }
            if (!inputs[1]) {
                const platform = yield this.env.prompt({
                    type: 'input',
                    name: 'platform',
                    message: `What platform would you like to ${inputs[0]} (${['android', 'ios'].map(v => chalk_1.default.green(v)).join(', ')}):`,
                });
                inputs[1] = platform.trim();
            }
            cli_framework_1.validate(inputs[1], 'platform', [cli_framework_1.validators.required]);
        });
    }
    run(inputs, options, runinfo) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { getPlatforms } = yield Promise.resolve().then(() => require('../../lib/integrations/cordova/project'));
            const { filterArgumentsForCordova } = yield Promise.resolve().then(() => require('../../lib/integrations/cordova/utils'));
            if (!this.project) {
                throw new errors_1.FatalException(`Cannot run ${chalk_1.default.green('ionic cordova platform')} outside a project directory.`);
            }
            const [action, platformName] = inputs;
            const platforms = yield getPlatforms(this.integration.root);
            if (action === 'add' && platforms.includes(platformName)) {
                this.env.log.msg(`Platform ${platformName} already exists.`);
                return;
            }
            const metadata = yield this.getMetadata();
            const cordovaArgs = filterArgumentsForCordova(metadata, options);
            if ((action === 'add' || action === 'remove') &&
                (options['save'] !== false && !options['nosave']) &&
                lodash.intersection(options['--'] || [], ['--save', '--nosave', '--no-save']).length === 0) {
                cordovaArgs.push('--save');
            }
            yield this.runCordova(cordovaArgs, {});
            const isLoggedIn = this.env.session.isLoggedIn();
            if (isLoggedIn && action === 'add' && options['resources'] && ['ios', 'android'].includes(platformName)) {
                yield executor_1.runCommand(runinfo, ['cordova', 'resources', platformName, '--force']);
            }
        });
    }
}
exports.PlatformCommand = PlatformCommand;
