"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const cli_framework_1 = require("@ionic/cli-framework");
const chalk_1 = require("chalk");
const lodash = require("lodash");
const errors_1 = require("../../lib/errors");
const utils_1 = require("../../lib/integrations/cordova/utils");
const base_1 = require("./base");
class PluginCommand extends base_1.CordovaCommand {
    getMetadata() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return {
                name: 'plugin',
                type: 'project',
                summary: 'Manage Cordova plugins',
                description: `
Like running ${chalk_1.default.green('cordova plugin')} directly, but provides friendly checks.
      `,
                exampleCommands: ['', 'add cordova-plugin-inappbrowser@latest', 'add phonegap-plugin-push --variable SENDER_ID=XXXXX', 'rm cordova-plugin-camera'],
                inputs: [
                    {
                        name: 'action',
                        summary: `${chalk_1.default.green('add')} or ${chalk_1.default.green('remove')} a plugin; ${chalk_1.default.green('ls')} or ${chalk_1.default.green('save')} all project plugins`,
                    },
                    {
                        name: 'plugin',
                        summary: `The name of the plugin (corresponds to ${chalk_1.default.green('add')} and ${chalk_1.default.green('remove')})`,
                    },
                ],
                options: [
                    {
                        name: 'force',
                        summary: `Force overwrite the plugin if it exists (corresponds to ${chalk_1.default.green('add')})`,
                        type: Boolean,
                        groups: [cli_framework_1.OptionGroup.Advanced, 'cordova', 'cordova-cli'],
                    },
                    {
                        name: 'variable',
                        summary: 'Specify plugin variables',
                        groups: ['cordova', 'cordova-cli'],
                        spec: { value: 'KEY=VALUE' },
                    },
                ],
            };
        });
    }
    preRun(inputs, options, runinfo) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            yield this.preRunChecks(runinfo);
            inputs[0] = !inputs[0] ? 'ls' : inputs[0];
            inputs[0] = (inputs[0] === 'rm') ? 'remove' : inputs[0];
            inputs[0] = (inputs[0] === 'list') ? 'ls' : inputs[0];
            cli_framework_1.validate(inputs[0], 'action', [cli_framework_1.contains(['add', 'remove', 'ls', 'save'], {})]);
            // If the action is list then lets just end here.
            if (['ls', 'save'].includes(inputs[0])) {
                yield this.runCordova(['plugin', inputs[0]], {});
                throw new errors_1.FatalException('', 0);
            }
            if (!inputs[1]) {
                const plugin = yield this.env.prompt({
                    message: `What plugin would you like to ${inputs[0]}:`,
                    type: 'input',
                    name: 'plugin',
                });
                inputs[1] = plugin;
            }
            cli_framework_1.validate(inputs[1], 'plugin', [cli_framework_1.validators.required]);
        });
    }
    run(inputs, options) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const [action] = inputs;
            const metadata = yield this.getMetadata();
            const cordovaArgs = utils_1.filterArgumentsForCordova(metadata, options);
            if ((action === 'add' || action === 'remove') &&
                (options['save'] !== false && !options['nosave']) &&
                lodash.intersection(options['--'] || [], ['--save', '--nosave', '--no-save']).length === 0) {
                cordovaArgs.push('--save');
            }
            yield this.runCordova(cordovaArgs, {});
        });
    }
}
exports.PluginCommand = PluginCommand;
