"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const chalk_1 = require("chalk");
const build_1 = require("../../lib/build");
const errors_1 = require("../../lib/errors");
const utils_1 = require("../../lib/integrations/cordova/utils");
const base_1 = require("./base");
class PrepareCommand extends base_1.CordovaCommand {
    getMetadata() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const options = [
                {
                    name: 'build',
                    summary: 'Do not invoke an Ionic build',
                    type: Boolean,
                    default: true,
                },
            ];
            const footnotes = [];
            const runner = this.project && (yield this.project.getBuildRunner());
            if (runner) {
                const libmetadata = yield runner.getCommandMetadata();
                options.push(...libmetadata.options || []);
                footnotes.push(...libmetadata.footnotes || []);
            }
            return {
                name: 'prepare',
                type: 'project',
                summary: 'Copies assets to Cordova platforms, preparing them for native builds',
                description: `
${chalk_1.default.green('ionic cordova prepare')} will do the following:

- Perform an Ionic build, which compiles web assets to ${chalk_1.default.bold('www/')}.
- Copy the ${chalk_1.default.bold('www/')} directory into your Cordova platforms.
- Transform ${chalk_1.default.bold('config.xml')} into platform-specific manifest files.
- Copy icons and splash screens from ${chalk_1.default.bold('resources/')} to into your Cordova platforms.
- Copy plugin files into specified platforms.

You may wish to use ${chalk_1.default.green('ionic cordova prepare')} if you run your project with Android Studio or Xcode.
      `,
                footnotes,
                exampleCommands: ['', 'ios', 'android'],
                inputs: [
                    {
                        name: 'platform',
                        summary: `The platform you would like to prepare (e.g. ${['android', 'ios'].map(v => chalk_1.default.green(v)).join(', ')})`,
                    },
                ],
                options,
            };
        });
    }
    preRun(inputs, options, runinfo) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            yield this.preRunChecks(runinfo);
        });
    }
    run(inputs, options) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { getPlatforms } = yield Promise.resolve().then(() => require('../../lib/integrations/cordova/project'));
            const [platform] = inputs;
            if (!this.project) {
                throw new errors_1.FatalException(`Cannot run ${chalk_1.default.green('ionic cordova prepare')} outside a project directory.`);
            }
            if (platform) {
                yield this.checkForPlatformInstallation(platform, {
                    promptToInstall: true,
                    promptToInstallRefusalMsg: (`Cannot prepare for ${chalk_1.default.green(platform)} unless the platform is installed.\n` +
                        `Did you mean just ${chalk_1.default.green('ionic cordova prepare')}?\n`),
                });
            }
            else {
                const platforms = yield getPlatforms(this.integration.root);
                if (platforms.length === 0) {
                    this.env.log.warn(`No platforms added to this project. Cannot prepare native platforms without any installed.\n` +
                        `Run ${chalk_1.default.green('ionic cordova platform add <platform>')} to add native platforms.`);
                    throw new errors_1.FatalException('', 0);
                }
            }
            const metadata = yield this.getMetadata();
            if (options.build) {
                const buildOptions = utils_1.generateOptionsForCordovaBuild(metadata, inputs, options);
                if (buildOptions['platform']) {
                    // TODO: use runner directly
                    yield build_1.build({ config: this.env.config, log: this.env.log, shell: this.env.shell, prompt: this.env.prompt, project: this.project }, inputs, buildOptions);
                }
                else {
                    this.env.log.warn(`Cannot perform Ionic build without ${chalk_1.default.green('platform')}. Falling back to just ${chalk_1.default.green('cordova prepare')}.\n` +
                        `Please supply a ${chalk_1.default.green('platform')} (e.g. ${['android', 'ios'].map(v => chalk_1.default.green(v)).join(', ')}) so the Ionic CLI can build web assets. The ${chalk_1.default.green('--no-build')} option will hide this warning.`);
                    this.env.log.nl();
                }
            }
            yield this.runCordova(utils_1.filterArgumentsForCordova(metadata, options), {});
        });
    }
}
exports.PrepareCommand = PrepareCommand;
