"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const chalk_1 = require("chalk");
const guards_1 = require("../../guards");
const errors_1 = require("../../lib/errors");
const utils_1 = require("../../lib/integrations/cordova/utils");
const base_1 = require("./base");
class RequirementsCommand extends base_1.CordovaCommand {
    getMetadata() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return {
                name: 'requirements',
                type: 'project',
                summary: 'Checks and print out all the requirements for platforms',
                description: `
Like running ${chalk_1.default.green('cordova requirements')} directly, but provides friendly checks.
      `,
                inputs: [
                    {
                        name: 'platform',
                        summary: `The platform for which you would like to gather requirements (${['android', 'ios'].map(v => chalk_1.default.green(v)).join(', ')})`,
                    },
                ],
            };
        });
    }
    preRun(inputs, options, runinfo) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            yield this.preRunChecks(runinfo);
        });
    }
    run(inputs, options) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const [platform] = inputs;
            if (!this.project) {
                throw new errors_1.FatalException(`Cannot run ${chalk_1.default.green('ionic cordova requirements')} outside a project directory.`);
            }
            yield this.checkForPlatformInstallation(platform, {
                promptToInstall: true,
                promptToInstallRefusalMsg: (`Can't gather requirements for ${chalk_1.default.green(platform)} unless the platform is installed.\n` +
                    `Did you mean just ${chalk_1.default.green('ionic cordova requirements')}?\n`),
            });
            const metadata = yield this.getMetadata();
            try {
                yield this.runCordova(utils_1.filterArgumentsForCordova(metadata, options), { showError: false, fatalOnError: false });
            }
            catch (e) {
                if (e.fatal || !guards_1.isExitCodeException(e)) {
                    throw e;
                }
                throw new errors_1.FatalException();
            }
        });
    }
}
exports.RequirementsCommand = RequirementsCommand;
