"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const cli_framework_1 = require("@ionic/cli-framework");
const format_1 = require("@ionic/cli-framework/utils/format");
const utils_fs_1 = require("@ionic/utils-fs");
const chalk_1 = require("chalk");
const Debug = require("debug");
const errors_1 = require("../../lib/errors");
const logger_1 = require("../../lib/utils/logger");
const npm_1 = require("../../lib/utils/npm");
const base_1 = require("./base");
const debug = Debug('ionic:commands:cordova:resources');
const AVAILABLE_RESOURCE_TYPES = ['icon', 'splash'];
class ResourcesCommand extends base_1.CordovaCommand {
    getMetadata() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return {
                name: 'resources',
                type: 'project',
                summary: 'Automatically create icon and splash screen resources',
                description: `
Ionic can automatically generate perfectly sized icons and splash screens from source images (${chalk_1.default.bold('.png')}, ${chalk_1.default.bold('.psd')}, or ${chalk_1.default.bold('.ai')}) for your Cordova platforms.

The source image for icons should ideally be at least ${chalk_1.default.bold('1024×1024px')} and located at ${chalk_1.default.bold('resources/icon.png')}. The source image for splash screens should ideally be at least ${chalk_1.default.bold('2732×2732px')} and located at ${chalk_1.default.bold('resources/splash.png')}. If you used ${chalk_1.default.green('ionic start')}, there should already be default Ionic resources in the ${chalk_1.default.bold('resources/')} directory, which you can overwrite.

You can also generate platform-specific icons and splash screens by placing them in the respective ${chalk_1.default.bold('resources/<platform>/')} directory. For example, to generate an icon for Android, place your image at ${chalk_1.default.bold('resources/android/icon.png')}.

By default, this command will not regenerate resources whose source image has not changed. To disable this functionality and always overwrite generated images, use ${chalk_1.default.green('--force')}.

For best results, the splash screen's artwork should roughly fit within a square (${chalk_1.default.bold('1200×1200px')}) at the center of the image. You can use ${chalk_1.default.bold('https://code.ionicframework.com/resources/splash.psd')} as a template for your splash screen.

${chalk_1.default.green('ionic cordova resources')} will automatically update your ${chalk_1.default.bold('config.xml')} to reflect the changes in the generated images, which Cordova then configures.

Cordova reference documentation:
- Icons: ${chalk_1.default.bold('https://cordova.apache.org/docs/en/latest/config_ref/images.html')}
- Splash Screens: ${chalk_1.default.bold('https://cordova.apache.org/docs/en/latest/reference/cordova-plugin-splashscreen/')}

This command uses Ionic servers, so we require you to be logged into your free Ionic account. Use ${chalk_1.default.green('ionic login')} to login.
      `,
                exampleCommands: ['', 'ios', 'android'],
                inputs: [
                    {
                        name: 'platform',
                        summary: `The platform for which you would like to generate resources (${['android', 'ios'].map(v => chalk_1.default.green(v)).join(', ')})`,
                    },
                ],
                options: [
                    {
                        name: 'force',
                        summary: 'Force regeneration of resources',
                        type: Boolean,
                        aliases: ['f'],
                    },
                    {
                        name: 'icon',
                        summary: 'Generate icon resources',
                        type: Boolean,
                        aliases: ['i'],
                    },
                    {
                        name: 'splash',
                        summary: 'Generate splash screen resources',
                        type: Boolean,
                        aliases: ['s'],
                    },
                    {
                        name: 'cordova-res',
                        summary: `Use ${chalk_1.default.green('cordova-res')} instead of Ionic resource server`,
                        type: Boolean,
                        groups: [cli_framework_1.OptionGroup.Hidden],
                    },
                ],
            };
        });
    }
    preRun(inputs, options, runinfo) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            yield this.preRunChecks(runinfo);
            const { promptToLogin } = yield Promise.resolve().then(() => require('../../lib/session'));
            const isLoggedIn = this.env.session.isLoggedIn();
            if (!options['cordova-res'] && !isLoggedIn) {
                this.env.log.warn(`You need to be logged into your Ionic account in order to run ${chalk_1.default.green(`ionic cordova resources`)}.\n`);
                yield promptToLogin(this.env);
            }
        });
    }
    getBuildPlatforms() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { getPlatforms } = yield Promise.resolve().then(() => require('../../lib/integrations/cordova/project'));
            const { RESOURCES } = yield Promise.resolve().then(() => require('../../lib/integrations/cordova/resources'));
            debug(`RESOURCES=${Object.keys(RESOURCES).length}`);
            const installedPlatforms = yield getPlatforms(this.integration.root);
            debug(`installedPlatforms=${installedPlatforms.map(e => chalk_1.default.bold(e)).join(', ')}`);
            const buildPlatforms = Object.keys(RESOURCES).filter(p => installedPlatforms.includes(p));
            debug(`buildPlatforms=${buildPlatforms.map(v => chalk_1.default.bold(v)).join(', ')}`);
            return buildPlatforms;
        });
    }
    run(inputs, options) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const platform = inputs[0] ? String(inputs[0]) : undefined;
            if (options['cordova-res']) {
                yield this.runCordovaRes(platform, options);
            }
            else {
                yield this.runResourceServer(platform, options);
            }
        });
    }
    runCordovaRes(platform, options) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            if (!this.project) {
                throw new errors_1.FatalException(`Cannot run ${chalk_1.default.green('ionic cordova resources')} outside a project directory.`);
            }
            const log = this.env.log.clone();
            log.handlers = logger_1.createDefaultLoggerHandlers(cli_framework_1.createPrefixedFormatter(chalk_1.default.dim(`[cordova-res]`)));
            const ws = log.createWriteStream(cli_framework_1.LOGGER_LEVELS.INFO);
            const args = [];
            if (platform) {
                args.push(platform);
            }
            if (options['icon']) {
                args.push('--type', 'icon');
            }
            else if (options['splash']) {
                args.push('--type', 'splash');
            }
            if (options['verbose']) {
                args.push('--verbose');
            }
            try {
                yield this.env.shell.run('cordova-res', args, { showCommand: true, fatalOnNotFound: false, cwd: this.project.directory, stream: ws });
            }
            catch (e) {
                if (e instanceof cli_framework_1.ShellCommandError && e.code === cli_framework_1.ERROR_SHELL_COMMAND_NOT_FOUND) {
                    const installArgs = yield npm_1.pkgManagerArgs(this.env.config.get('npmClient'), { command: 'install', pkg: 'cordova-res', global: true });
                    throw new errors_1.FatalException(`${chalk_1.default.green('cordova-res')} was not found on your PATH. Please install it globally:\n` +
                        `${chalk_1.default.green(installArgs.join(' '))}\n`);
                }
                throw e;
            }
        });
    }
    runResourceServer(platform, options) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { loadConfigXml } = yield Promise.resolve().then(() => require('../../lib/integrations/cordova/config'));
            const { addResourcesToConfigXml, createImgDestinationDirectories, findMostSpecificSourceImage, getImageResources, getSourceImages, transformResourceImage, uploadSourceImage } = yield Promise.resolve().then(() => require('../../lib/integrations/cordova/resources'));
            const { force } = options;
            const tasks = this.createTaskChain();
            // if no resource filters are passed as arguments assume to use all.
            let resourceTypes = AVAILABLE_RESOURCE_TYPES.filter((type, index, array) => options[type]);
            resourceTypes = resourceTypes.length ? resourceTypes : AVAILABLE_RESOURCE_TYPES;
            // await this.checkForPlatformInstallation(platform, { promptToInstall: true });
            const conf = yield loadConfigXml(this.integration);
            const buildPlatforms = platform ? [platform] : yield this.getBuildPlatforms();
            if (buildPlatforms.length === 0) {
                throw new errors_1.FatalException(`No platforms detected. Please run: ${chalk_1.default.green('ionic cordova platform add')}`);
            }
            tasks.next(`Collecting resource configuration and source images`);
            const orientation = conf.getPreference('Orientation') || 'default';
            // Convert the resource structure to a flat array then filter the array so
            // that it only has img resources that we need. Finally add src path to the
            // items that remain.
            let imgResources = getImageResources(this.integration.root)
                .filter(img => orientation === 'default' || typeof img.orientation === 'undefined' || img.orientation === orientation)
                .filter(img => buildPlatforms.includes(img.platform))
                .filter(img => resourceTypes.includes(img.resType));
            if (platform) {
                imgResources = imgResources.filter(img => img.platform === platform);
            }
            debug(`imgResources=${imgResources.length}`);
            // Create the resource directories that are needed for the images we will create
            const buildDirResponses = yield createImgDestinationDirectories(imgResources);
            debug(`${chalk_1.default.cyan('createImgDestinationDirectories')} completed: ${buildDirResponses.length}`);
            // Check /resources and /resources/<platform> directories for src files
            // Update imgResources to have their src attributes to equal the most
            // specific src img found
            let srcImagesAvailable = [];
            try {
                srcImagesAvailable = yield getSourceImages(this.integration.root, buildPlatforms, resourceTypes);
                debug(`${chalk_1.default.cyan('getSourceImages')} completed: (${srcImagesAvailable.map(v => chalk_1.default.bold(format_1.prettyPath(v.path))).join(', ')})`);
            }
            catch (e) {
                this.env.log.error(`Error in ${chalk_1.default.green('getSourceImages')}: ${e.stack ? e.stack : e}`);
            }
            imgResources = imgResources.map(img => {
                const mostSpecificImageAvailable = findMostSpecificSourceImage(img, srcImagesAvailable);
                return Object.assign({}, img, { imageId: mostSpecificImageAvailable && mostSpecificImageAvailable.imageId ? mostSpecificImageAvailable.imageId : undefined });
            });
            debug(`imgResources=${imgResources.length}`);
            // If there are any imgResources that have missing images then end
            // processing and inform the user
            const missingSrcImages = imgResources.filter(img => !img.imageId);
            if (missingSrcImages.length > 0) {
                const missingImageText = missingSrcImages
                    .reduce((list, img) => {
                    const str = `${img.platform}/${img.resType}`;
                    if (!list.includes(str)) {
                        list.push(str);
                    }
                    return list;
                }, [])
                    .map(v => `- ${chalk_1.default.bold(v)}`)
                    .join('\n');
                throw new errors_1.FatalException(`Source image files were not found for the following platforms/types:\n${missingImageText}\n\n` +
                    `Please review ${chalk_1.default.green('--help')}`);
            }
            tasks.next(`Filtering out image resources that do not need regeneration`);
            const cachedSourceIds = srcImagesAvailable
                .filter(img => img.imageId && img.cachedId && img.imageId === img.cachedId)
                .map(img => img.imageId);
            if (!force) {
                const keepImgResources = yield Promise.all(imgResources.map((img) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    if (!(yield utils_fs_1.pathExists(img.dest))) {
                        return true;
                    }
                    return img.imageId && !cachedSourceIds.includes(img.imageId);
                })));
                imgResources = imgResources.filter((img, i) => keepImgResources[i]);
                if (imgResources.length === 0) {
                    tasks.end();
                    this.env.log.nl();
                    this.env.log.info('No need to regenerate images.\n' +
                        'This could mean your generated images exist and do not need updating or your source files are unchanged.\n\n' +
                        `You can force image regeneration with the ${chalk_1.default.green('--force')} option.`);
                    throw new errors_1.FatalException('', 0);
                }
            }
            const uploadTask = tasks.next(`Uploading source images to prepare for transformations`);
            let count = 0;
            // Upload images to service to prepare for resource transformations
            const imageUploadResponses = yield Promise.all(srcImagesAvailable.map((srcImage) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                const response = yield uploadSourceImage(this.env, srcImage);
                count += 1;
                uploadTask.msg = `Uploading source images to prepare for transformations: ${chalk_1.default.bold(`${count} / ${srcImagesAvailable.length}`)} complete`;
                return response;
            })));
            debug(`${chalk_1.default.cyan('uploadSourceImages')} completed: responses=%O`, imageUploadResponses);
            srcImagesAvailable = srcImagesAvailable.map((img, index) => {
                return Object.assign({}, img, { width: imageUploadResponses[index].Width, height: imageUploadResponses[index].Height, vector: imageUploadResponses[index].Vector });
            });
            debug('srcImagesAvailable=%O', srcImagesAvailable);
            // If any images are asking to be generated but are not of the correct size
            // inform the user and continue on.
            const imagesTooLargeForSource = imgResources.filter(img => {
                const resourceSourceImage = srcImagesAvailable.find(srcImage => srcImage.imageId === img.imageId);
                if (!resourceSourceImage) {
                    return true;
                }
                return !resourceSourceImage.vector && (img.width > resourceSourceImage.width || img.height > resourceSourceImage.height);
            });
            debug('imagesTooLargeForSource=%O', imagesTooLargeForSource);
            // Remove all images too large for transformations
            imgResources = imgResources.filter(img => {
                return !imagesTooLargeForSource.find(tooLargeForSourceImage => img.name === tooLargeForSourceImage.name);
            });
            if (imgResources.length === 0) {
                tasks.end();
                this.env.log.nl();
                this.env.log.info('No need to regenerate images--images too large for transformation.'); // TODO: improve messaging
                throw new errors_1.FatalException('', 0);
            }
            // Call the transform service and output images to appropriate destination
            const generateTask = tasks.next(`Generating platform resources`);
            count = 0;
            const transforms = imgResources.map((img) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                const result = yield transformResourceImage(this.env, img);
                count += 1;
                generateTask.msg = `Generating platform resources: ${chalk_1.default.bold(`${count} / ${imgResources.length}`)} complete`;
                return result;
            }));
            const transformResults = yield Promise.all(transforms);
            generateTask.msg = `Generating platform resources: ${chalk_1.default.bold(`${imgResources.length} / ${imgResources.length}`)} complete`;
            debug('transforms completed');
            const transformErrors = transformResults.map(result => result.error).filter((err) => typeof err !== 'undefined');
            if (transformErrors.length > 0) {
                throw new errors_1.FatalException(`Encountered ${transformErrors.length} error(s) during image transforms:\n\n` +
                    transformErrors.map((err, i) => `${i + 1}): ` + chalk_1.default.red(err.toString())).join('\n\n'));
            }
            yield Promise.all(transformResults.map((result) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                yield utils_fs_1.copy(result.tmpDest, result.resource.dest);
                debug('copied transformed image %s into project as %s', result.tmpDest, result.resource.dest);
            })));
            yield Promise.all(srcImagesAvailable.map((img) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                yield utils_fs_1.cacheFileChecksum(img.path, img.imageId);
            })));
            tasks.next(`Modifying config.xml to add new image resources`);
            const imageResourcesForConfig = imgResources.reduce((rc, img) => {
                if (!rc[img.platform]) {
                    rc[img.platform] = {
                        [img.resType]: {
                            images: [],
                            nodeName: '',
                            nodeAttributes: [],
                        },
                    };
                }
                if (!rc[img.platform][img.resType]) {
                    rc[img.platform][img.resType] = {
                        images: [],
                        nodeName: '',
                        nodeAttributes: [],
                    };
                }
                rc[img.platform][img.resType].images.push({
                    name: img.name,
                    width: img.width,
                    height: img.height,
                    density: img.density,
                });
                rc[img.platform][img.resType].nodeName = img.nodeName;
                rc[img.platform][img.resType].nodeAttributes = img.nodeAttributes;
                return rc;
            }, {});
            const platformList = Object.keys(imageResourcesForConfig);
            yield addResourcesToConfigXml(conf, platformList, imageResourcesForConfig);
            tasks.end();
            // All images that were not processed
            if (imagesTooLargeForSource.length > 0) {
                const imagesTooLargeForSourceMsg = imagesTooLargeForSource
                    .map(img => `    ${chalk_1.default.bold(img.name)}     ${img.platform}/${img.resType} needed ${img.width}×${img.height}px`)
                    .concat((imagesTooLargeForSource.length > 0) ? `\nThe following images were not created because their source image was too small:` : [])
                    .reverse();
                this.env.log.rawmsg(imagesTooLargeForSourceMsg.join('\n'));
            }
            yield conf.save();
        });
    }
}
exports.ResourcesCommand = ResourcesCommand;
