"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const cli_framework_1 = require("@ionic/cli-framework");
const process_1 = require("@ionic/cli-framework/utils/process");
const chalk_1 = require("chalk");
const path = require("path");
const url = require("url");
const build_1 = require("../../lib/build");
const errors_1 = require("../../lib/errors");
const config_1 = require("../../lib/integrations/cordova/config");
const utils_1 = require("../../lib/integrations/cordova/utils");
const serve_1 = require("../../lib/serve");
const logger_1 = require("../../lib/utils/logger");
const npm_1 = require("../../lib/utils/npm");
const base_1 = require("./base");
const CORDOVA_ANDROID_PACKAGE_PATH = 'platforms/android/app/build/outputs/apk/';
const CORDOVA_IOS_SIMULATOR_PACKAGE_PATH = 'platforms/ios/build/emulator';
const CORDOVA_IOS_DEVICE_PACKAGE_PATH = 'platforms/ios/build/device';
const NATIVE_RUN_OPTIONS = [
    {
        name: 'native-run',
        summary: `Use ${chalk_1.default.green('native-run')} instead of Cordova for running the app`,
        type: Boolean,
        groups: [cli_framework_1.OptionGroup.Hidden, 'native-run'],
        hint: chalk_1.default.dim('[native-run]'),
    },
    {
        name: 'connect',
        summary: 'Do not tie the running app to the process',
        type: Boolean,
        default: true,
        groups: [cli_framework_1.OptionGroup.Hidden, 'native-run'],
        hint: chalk_1.default.dim('[native-run]'),
    },
    {
        name: 'json',
        summary: `Output ${chalk_1.default.green('--list')} targets in JSON`,
        type: Boolean,
        groups: [cli_framework_1.OptionGroup.Hidden, 'native-run'],
        hint: chalk_1.default.dim('[native-run]'),
    },
];
class RunCommand extends base_1.CordovaCommand {
    getMetadata() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            let groups = [];
            const exampleCommands = [
                ...base_1.CORDOVA_BUILD_EXAMPLE_COMMANDS,
                'android -l',
                'ios --livereload',
                'ios --livereload-url=http://localhost:8100',
            ].sort();
            const options = [
                {
                    name: 'list',
                    summary: 'List all available targets',
                    type: Boolean,
                    groups: ['cordova', 'cordova-cli', 'native-run'],
                },
                // Build Options
                {
                    name: 'build',
                    summary: 'Do not invoke Ionic build',
                    type: Boolean,
                    default: true,
                },
                ...build_1.COMMON_BUILD_COMMAND_OPTIONS.filter(o => !['engine', 'platform'].includes(o.name)),
                // Serve Options
                ...serve_1.COMMON_SERVE_COMMAND_OPTIONS.filter(o => !['livereload'].includes(o.name)),
                {
                    name: 'livereload',
                    summary: 'Spin up dev server to live-reload www files',
                    type: Boolean,
                    aliases: ['l'],
                },
                {
                    name: 'livereload-url',
                    summary: 'Provide a custom URL to the dev server',
                },
            ];
            const footnotes = [
                {
                    id: 'remote-debugging-docs',
                    url: 'https://ionicframework.com/docs/developer-resources/developer-tips',
                    shortUrl: 'https://ion.link/remote-debugging-docs',
                },
            ];
            const serveRunner = this.project && (yield this.project.getServeRunner());
            const buildRunner = this.project && (yield this.project.getBuildRunner());
            if (buildRunner) {
                const libmetadata = yield buildRunner.getCommandMetadata();
                groups = libmetadata.groups || [];
                options.push(...libmetadata.options || []);
                footnotes.push(...libmetadata.footnotes || []);
            }
            if (serveRunner) {
                const libmetadata = yield serveRunner.getCommandMetadata();
                const existingOpts = options.map(o => o.name);
                groups = libmetadata.groups || [];
                options.push(...(libmetadata.options || [])
                    .filter(o => !existingOpts.includes(o.name) && o.groups && o.groups.includes('cordova'))
                    .map(o => (Object.assign({}, o, { hint: `${o.hint ? `${o.hint} ` : ''}${chalk_1.default.dim('(--livereload)')}` }))));
                footnotes.push(...libmetadata.footnotes || []);
            }
            // Cordova Options
            options.push(...base_1.CORDOVA_RUN_OPTIONS);
            // `native-run` Options
            options.push(...NATIVE_RUN_OPTIONS);
            return {
                name: 'run',
                type: 'project',
                summary: 'Run an Ionic project on a connected device',
                description: `
Like running ${chalk_1.default.green('cordova run')} or ${chalk_1.default.green('cordova emulate')} directly, but performs ${chalk_1.default.green('ionic build')} before deploying to the device or emulator. Optionally specify the ${chalk_1.default.green('--livereload')} option to use the dev server from ${chalk_1.default.green('ionic serve')} for livereload functionality.

For Android and iOS, you can setup Remote Debugging on your device with browser development tools using these docs[^remote-debugging-docs].

Just like with ${chalk_1.default.green('ionic cordova build')}, you can pass additional options to the Cordova CLI using the ${chalk_1.default.green('--')} separator. To pass additional options to the dev server, consider using ${chalk_1.default.green('ionic serve')} and the ${chalk_1.default.green('--livereload-url')} option.
      `,
                footnotes,
                exampleCommands,
                inputs: [
                    {
                        name: 'platform',
                        summary: `The platform to run (e.g. ${['android', 'ios'].map(v => chalk_1.default.green(v)).join(', ')})`,
                    },
                ],
                options,
                groups,
            };
        });
    }
    preRun(inputs, options, runinfo) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            yield this.preRunChecks(runinfo);
            const metadata = yield this.getMetadata();
            if (options['noproxy']) {
                this.env.log.warn(`The ${chalk_1.default.green('--noproxy')} option has been deprecated. Please use ${chalk_1.default.green('--no-proxy')}.`);
                options['proxy'] = false;
            }
            if (options['x']) {
                options['proxy'] = false;
            }
            if (options['livereload-url']) {
                options['livereload'] = true;
            }
            if (!options['build'] && options['livereload']) {
                this.env.log.warn(`No livereload with ${chalk_1.default.green('--no-build')}.`);
                options['livereload'] = false;
            }
            if (options['list']) {
                if (!options['device'] && !options['emulator']) {
                    if (metadata.name === 'emulate') {
                        options['emulator'] = true;
                    }
                }
                if (options['native-run']) {
                    const args = createNativeRunListArgs(inputs, options);
                    yield this.nativeRun(args);
                }
                else {
                    const args = utils_1.filterArgumentsForCordova(metadata, options);
                    yield this.runCordova(['run', ...args.slice(1)], {});
                }
                throw new errors_1.FatalException('', 0);
            }
            if (!inputs[0]) {
                const platform = yield this.env.prompt({
                    type: 'input',
                    name: 'platform',
                    message: `What platform would you like to run (${['android', 'ios'].map(v => chalk_1.default.green(v)).join(', ')}):`,
                });
                inputs[0] = platform.trim();
            }
            yield this.checkForPlatformInstallation(inputs[0]);
        });
    }
    run(inputs, options) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            if (!this.project) {
                throw new errors_1.FatalException(`Cannot run ${chalk_1.default.green('ionic cordova run/emulate')} outside a project directory.`);
            }
            const metadata = yield this.getMetadata();
            if (options['livereload']) {
                let livereloadUrl = options['livereload-url'] ? String(options['livereload-url']) : undefined;
                if (!livereloadUrl) {
                    // TODO: use runner directly
                    const details = yield serve_1.serve({ flags: this.env.flags, config: this.env.config, log: this.env.log, prompt: this.env.prompt, shell: this.env.shell, project: this.project }, inputs, utils_1.generateOptionsForCordovaBuild(metadata, inputs, options));
                    if (details.externallyAccessible === false && !options['native-run']) {
                        const extra = serve_1.LOCAL_ADDRESSES.includes(details.externalAddress) ? '\nEnsure you have proper port forwarding setup from your device to your computer.' : '';
                        this.env.log.warn(`Your device or emulator may not be able to access ${chalk_1.default.bold(details.externalAddress)}.${extra}\n\n`);
                    }
                    livereloadUrl = `${details.protocol || 'http'}://${options['native-run'] ? details.localAddress : details.externalAddress}:${details.port}`;
                }
                const conf = yield config_1.loadConfigXml(this.integration);
                process_1.onBeforeExit(() => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    conf.resetContentSrc();
                    yield conf.save();
                }));
                conf.writeContentSrc(livereloadUrl);
                yield conf.save();
                const cordovalog = this.env.log.clone();
                cordovalog.handlers = logger_1.createDefaultLoggerHandlers(cli_framework_1.createPrefixedFormatter(`${chalk_1.default.dim(`[cordova]`)} `));
                const cordovalogws = cordovalog.createWriteStream(cli_framework_1.LOGGER_LEVELS.INFO);
                if (options['native-run']) {
                    // hack to do just Cordova build instead
                    metadata.name = 'build';
                    const buildOpts = { stream: cordovalogws };
                    // ignore very verbose compiler output unless --verbose (still pipe stderr)
                    if (!options['verbose']) {
                        buildOpts.stdio = ['ignore', 'ignore', 'pipe'];
                    }
                    yield this.runCordova(utils_1.filterArgumentsForCordova(metadata, options), buildOpts);
                    const platform = inputs[0];
                    const packagePath = getPackagePath(conf.getProjectInfo().name, platform, options['emulator']);
                    const nativeRunArgs = createNativeRunArgs(packagePath, platform, livereloadUrl, options);
                    yield this.nativeRun(nativeRunArgs);
                }
                else {
                    yield this.runCordova(utils_1.filterArgumentsForCordova(metadata, options), { stream: cordovalogws });
                    yield process_1.sleepForever();
                }
            }
            else {
                if (options.build) {
                    // TODO: use runner directly
                    yield build_1.build({ config: this.env.config, log: this.env.log, shell: this.env.shell, prompt: this.env.prompt, project: this.project }, inputs, utils_1.generateOptionsForCordovaBuild(metadata, inputs, options));
                }
                yield this.runCordova(utils_1.filterArgumentsForCordova(metadata, options));
            }
        });
    }
    nativeRun(args) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            if (!this.project) {
                throw new errors_1.FatalException(`Cannot run ${chalk_1.default.green('ionic cordova run/emulate')} outside a project directory.`);
            }
            let ws;
            if (!args.includes('--list')) {
                const log = this.env.log.clone();
                log.handlers = logger_1.createDefaultLoggerHandlers(cli_framework_1.createPrefixedFormatter(chalk_1.default.dim(`[native-run]`)));
                ws = log.createWriteStream(cli_framework_1.LOGGER_LEVELS.INFO);
            }
            try {
                yield this.env.shell.run('native-run', args, { showCommand: !args.includes('--json'), fatalOnNotFound: false, cwd: this.project.directory, stream: ws });
            }
            catch (e) {
                if (e instanceof cli_framework_1.ShellCommandError && e.code === cli_framework_1.ERROR_SHELL_COMMAND_NOT_FOUND) {
                    const cdvInstallArgs = yield npm_1.pkgManagerArgs(this.env.config.get('npmClient'), { command: 'install', pkg: 'native-run', global: true });
                    throw new errors_1.FatalException(`${chalk_1.default.green('native-run')} was not found on your PATH. Please install it globally:\n` +
                        `${chalk_1.default.green(cdvInstallArgs.join(' '))}\n`);
                }
                throw e;
            }
            // If we connect the `native-run` process to the running app, then we
            // should also connect the Ionic CLI with the running `native-run` process.
            // This will exit the Ionic CLI when `native-run` exits.
            if (args.includes('--connect')) {
                process_1.processExit(0); // tslint:disable-line:no-floating-promises
            }
        });
    }
}
exports.RunCommand = RunCommand;
function createNativeRunArgs(packagePath, platform, livereloadUrl, options) {
    const opts = [platform, '--app', packagePath];
    const target = options['target'] ? String(options['target']) : undefined;
    if (target) {
        opts.push('--target', target);
    }
    else if (options['emulator']) {
        opts.push('--virtual');
    }
    if (options['connect']) {
        opts.push('--connect');
    }
    if (!options['livereload-url']) {
        const { port } = url.parse(livereloadUrl);
        opts.push('--forward', `${port}:${port}`);
    }
    if (options['json']) {
        opts.push('--json');
    }
    if (options['verbose']) {
        opts.push('--verbose');
    }
    return opts;
}
function createNativeRunListArgs(inputs, options) {
    const args = [];
    if (inputs[0]) {
        args.push(inputs[0]);
    }
    args.push('--list');
    if (options['json']) {
        args.push('--json');
    }
    if (options['device']) {
        args.push('--device');
    }
    if (options['emulator']) {
        args.push('--virtual');
    }
    if (options['json']) {
        args.push('--json');
    }
    return args;
}
function getPackagePath(appName, platform, emulator) {
    if (platform === 'android') {
        // TODO: don't hardcode this/support multiple build paths (ex: multiple arch builds)
        // use app/build/outputs/apk/debug/output.json?
        return path.join(CORDOVA_ANDROID_PACKAGE_PATH, 'debug', 'app-debug.apk');
    }
    if (platform === 'ios' && emulator) {
        return path.join(CORDOVA_IOS_SIMULATOR_PACKAGE_PATH, `${appName}.app`);
    }
    else {
        return path.join(CORDOVA_IOS_DEVICE_PACKAGE_PATH, `${appName}.ipa`);
    }
}
