"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const cli_framework_1 = require("@ionic/cli-framework");
const format_1 = require("@ionic/cli-framework/utils/format");
const process_1 = require("@ionic/cli-framework/utils/process");
const chalk_1 = require("chalk");
const Debug = require("debug");
const guards_1 = require("../../guards");
const command_1 = require("../../lib/command");
const errors_1 = require("../../lib/errors");
const debug = Debug('ionic:commands:deploy:build');
class BuildCommand extends command_1.Command {
    getMetadata() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const dashUrl = this.env.config.getDashUrl();
            return {
                name: 'build',
                type: 'project',
                summary: 'Create a deploy build on Appflow',
                description: `
This command creates a deploy build on Ionic Appflow. While the build is running, it prints the remote build log to the terminal.

Customizing the build:
- The ${chalk_1.default.green('--environment')} and ${chalk_1.default.green('--channel')} options can be used to customize the groups of values exposed to the build.

Apart from ${chalk_1.default.green('--commit')}, every option can be specified using the full name setup within the Appflow Dashboard[^dashboard].
`,
                footnotes: [
                    {
                        id: 'dashboard',
                        url: dashUrl,
                    },
                ],
                exampleCommands: [
                    '',
                    '--environment="My Custom Environment Name"',
                    '--commit=2345cd3305a1cf94de34e93b73a932f25baac77c',
                    '--channel="Master"',
                    '--channel="Master" --channel="My Custom Channel"',
                ],
                options: [
                    {
                        name: 'environment',
                        summary: 'The group of environment variables exposed to your build',
                        type: String,
                        spec: { value: 'name' },
                    },
                    {
                        name: 'channel',
                        summary: 'The channel you want to auto deploy the build to. This can be repeated multiple times if multiple channels need to be specified.',
                        type: String,
                        spec: { value: 'name' },
                    },
                    {
                        name: 'commit',
                        summary: 'Commit (defaults to HEAD)',
                        type: String,
                        groups: [cli_framework_1.OptionGroup.Advanced],
                        spec: { value: 'sha1' },
                    },
                ],
            };
        });
    }
    run(inputs, options) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            if (!this.project) {
                throw new errors_1.FatalException(`Cannot run ${chalk_1.default.green('ionic deploy build')} outside a project directory.`);
            }
            const token = this.env.session.getUserToken();
            const appflowId = yield this.project.requireAppflowId();
            if (!options.commit) {
                options.commit = (yield this.env.shell.output('git', ['rev-parse', 'HEAD'], { cwd: this.project.directory })).trim();
                debug(`Commit hash: ${chalk_1.default.bold(options.commit)}`);
            }
            let build = yield this.createDeployBuild(appflowId, token, options);
            const buildId = build.job_id;
            const details = format_1.columnar([
                ['Appflow ID', chalk_1.default.bold(appflowId)],
                ['Build ID', chalk_1.default.bold(buildId.toString())],
                ['Commit', chalk_1.default.bold(`${build.commit.sha.substring(0, 6)} ${build.commit.note}`)],
                ['Environment', build.environment_name ? chalk_1.default.bold(build.environment_name) : chalk_1.default.dim('not set')],
                ['Channels', build.pending_channels.length ? build.pending_channels.map(v => chalk_1.default.bold(`"${v}"`)).join(', ') : chalk_1.default.dim('not set')],
            ], { vsep: ':' });
            this.env.log.ok(`Build created\n` +
                details + '\n\n');
            build = yield this.tailBuildLog(appflowId, buildId, token);
            if (build.state !== 'success') {
                throw new Error('Build failed');
            }
        });
    }
    createDeployBuild(appflowId, token, options) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { req } = yield this.env.client.make('POST', `/apps/${appflowId}/deploys/verbose_post`);
            let channels = [];
            if (options.channel) {
                if (typeof (options.channel) === 'string') {
                    channels.push(String(options.channel));
                }
                else if (typeof (options.channel) === 'object') {
                    channels = channels.concat(options.channel);
                }
            }
            req.set('Authorization', `Bearer ${token}`).send({
                commit_sha: options.commit,
                environment_name: options.environment,
                channel_names: channels ? channels : undefined,
            });
            try {
                const res = yield this.env.client.do(req);
                return res.data;
            }
            catch (e) {
                if (guards_1.isSuperAgentError(e)) {
                    if (e.response.status === 401) {
                        this.env.log.error('Try logging out and back in again.');
                    }
                    const apiErrorMessage = (e.response.body.error && e.response.body.error.message) ? e.response.body.error.message : 'Api Error';
                    throw new errors_1.FatalException(`Unable to create build: ` + apiErrorMessage);
                }
                else {
                    throw e;
                }
            }
        });
    }
    tailBuildLog(appflowId, buildId, token) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            let build;
            let start = 0;
            const ws = this.env.log.createWriteStream(cli_framework_1.LOGGER_LEVELS.INFO, false);
            let isCreatedMessage = false;
            while (!(build && (build.state === 'success' || build.state === 'failed'))) {
                yield process_1.sleep(5000);
                build = yield this.getDeployBuild(appflowId, buildId, token);
                if (build && build.state === 'created' && !isCreatedMessage) {
                    ws.write(chalk_1.default.yellow('Concurrency limit reached: build will start as soon as other builds finish.'));
                    isCreatedMessage = true;
                }
                const trace = build.job.trace;
                if (trace.length > start) {
                    ws.write(trace.substring(start));
                    start = trace.length;
                }
            }
            ws.end();
            return build;
        });
    }
    getDeployBuild(appflowId, buildId, token) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { req } = yield this.env.client.make('GET', `/apps/${appflowId}/deploys/${buildId}`);
            req.set('Authorization', `Bearer ${token}`).send();
            try {
                const res = yield this.env.client.do(req);
                return res.data;
            }
            catch (e) {
                if (guards_1.isSuperAgentError(e)) {
                    if (e.response.status === 401) {
                        this.env.log.error('Try logging out and back in again.');
                    }
                    const apiErrorMessage = (e.response.body.error && e.response.body.error.message) ? e.response.body.error.message : 'Api Error';
                    throw new errors_1.FatalException(`Unable to get build ${buildId}: ` + apiErrorMessage);
                }
                else {
                    throw e;
                }
            }
        });
    }
}
exports.BuildCommand = BuildCommand;
