"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const cli_framework_1 = require("@ionic/cli-framework");
const array_1 = require("@ionic/cli-framework/utils/array");
const utils_fs_1 = require("@ionic/utils-fs");
const chalk_1 = require("chalk");
const crypto = require("crypto");
const fs = require("fs");
const path = require("path");
const command_1 = require("../../lib/command");
const errors_1 = require("../../lib/errors");
class DeployManifestCommand extends command_1.Command {
    getMetadata() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return {
                name: 'manifest',
                type: 'project',
                summary: 'Generates a manifest file for the deploy service from a built app directory',
                groups: [cli_framework_1.CommandGroup.Hidden],
            };
        });
    }
    run() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            if (!this.project) {
                throw new errors_1.FatalException(`Cannot run ${chalk_1.default.green('ionic deploy manifest')} outside a project directory.`);
            }
            const buildDir = path.resolve(this.project.directory, 'www'); // TODO: this is hard-coded
            const manifest = yield this.getFilesAndSizesAndHashesForGlobPattern(buildDir);
            yield utils_fs_1.writeFile(path.resolve(buildDir, 'pro-manifest.json'), JSON.stringify(manifest, undefined, 2), { encoding: 'utf8' });
        });
    }
    getFilesAndSizesAndHashesForGlobPattern(buildDir) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const contents = yield utils_fs_1.readdirp(buildDir);
            const stats = yield array_1.map(contents, (f) => tslib_1.__awaiter(this, void 0, void 0, function* () { return [f, yield utils_fs_1.stat(f)]; }));
            const files = stats.filter(([, stat]) => !stat.isDirectory());
            const items = yield Promise.all(files.map(([f, stat]) => this.getFileAndSizeAndHashForFile(buildDir, f, stat)));
            return items.filter(item => item.href !== 'pro-manifest.json');
        });
    }
    getFileAndSizeAndHashForFile(buildDir, file, stat) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const buffer = yield this.readFile(file);
            return {
                href: path.relative(buildDir, file),
                size: stat.size,
                integrity: this.getIntegrity(buffer),
            };
        });
    }
    readFile(file) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return new Promise((resolve, reject) => {
                fs.readFile(file, (err, buffer) => {
                    if (err) {
                        return reject(err);
                    }
                    resolve(buffer);
                });
            });
        });
    }
    getIntegrity(data) {
        return ['sha256', 'sha384', 'sha512']
            .map(algorithm => {
            const hash = crypto.createHash(algorithm);
            hash.update(data);
            return algorithm + '-' + hash.digest('base64');
        })
            .join(' ');
    }
}
exports.DeployManifestCommand = DeployManifestCommand;
