"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const array_1 = require("@ionic/cli-framework/utils/array");
const chalk_1 = require("chalk");
const Debug = require("debug");
const guards_1 = require("../../guards");
const command_1 = require("../../lib/command");
const errors_1 = require("../../lib/errors");
const debug = Debug('ionic:commands:doctor:base');
class DoctorCommand extends command_1.Command {
    getRegistry() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            if (!this.project) {
                throw new errors_1.FatalException(`Cannot use ${chalk_1.default.green('ionic doctor')} outside a project directory.`);
            }
            const { AilmentRegistry } = yield Promise.resolve().then(() => require('../../lib/doctor'));
            const registry = new AilmentRegistry();
            yield this.project.registerAilments(registry);
            return registry;
        });
    }
    detectAilments() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const registry = yield this.getRegistry();
            let count = 0;
            const tasks = this.createTaskChain();
            const isLoggedIn = this.env.session.isLoggedIn();
            if (!isLoggedIn) {
                this.env.log.warn(`For best results, please make sure you're logged in to Ionic.\nSome issues can't be detected without authentication. Run:\n\n    ${chalk_1.default.green('ionic login')}`);
            }
            const detectTask = tasks.next('Detecting issues');
            const ailments = registry.ailments.filter(ailment => {
                if (this.env.config.get(`doctor.issues.${ailment.id}.ignored`)) {
                    debug('Issue %s ignored by config', ailment.id);
                    return false;
                }
                if (!ailment.implicit) {
                    debug('Issue %s will not be implicitly detected', ailment.id);
                    return false;
                }
                return true;
            });
            const detectedAilments = yield array_1.concurrentFilter(ailments, (ailment) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                let detected = false;
                try {
                    detected = yield ailment.detected();
                    debug('Detected %s: %s', ailment.id, detected);
                }
                catch (e) {
                    this.env.log.error(`Error while checking ${chalk_1.default.bold(ailment.id)}:\n` +
                        `${chalk_1.default.red(e.stack ? e.stack : e)}`);
                }
                count++;
                detectTask.msg = `Detecting issues: ${chalk_1.default.bold(`${count} / ${ailments.length}`)} complete`;
                return detected;
            }));
            detectTask.msg = `Detecting issues: ${chalk_1.default.bold(`${ailments.length} / ${ailments.length}`)} complete`;
            tasks.end();
            return detectedAilments;
        });
    }
    detectTreatableAilments() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const ailments = yield this.detectAilments();
            return ailments.filter((ailment) => guards_1.isTreatableAilment(ailment));
        });
    }
}
exports.DoctorCommand = DoctorCommand;
