"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const cli_framework_1 = require("@ionic/cli-framework");
const chalk_1 = require("chalk");
const guards_1 = require("../../guards");
const errors_1 = require("../../lib/errors");
const base_1 = require("./base");
class DoctorCheckCommand extends base_1.DoctorCommand {
    getMetadata() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return {
                name: 'check',
                type: 'project',
                summary: 'Check the health of your Ionic project',
                description: `
This command detects and prints common issues and suggested steps to fix them.

Some issues can be fixed automatically. See ${chalk_1.default.green('ionic doctor treat --help')}.

Optionally supply the ${chalk_1.default.green('id')} argument to check a single issue. Use ${chalk_1.default.green('ionic doctor list')} to list all known issues.
      `,
                exampleCommands: [
                    '',
                    'git-not-used',
                ],
                inputs: [
                    {
                        name: 'id',
                        summary: 'The issue identifier',
                    },
                ],
            };
        });
    }
    run(inputs, options) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const [id] = inputs;
            if (id) {
                const registry = yield this.getRegistry();
                const ailmentIds = registry.ailments.map(a => a.id);
                cli_framework_1.validate(id, 'id', [cli_framework_1.contains(ailmentIds, {})]);
                const ailment = registry.get(id);
                if (!ailment) {
                    throw new errors_1.FatalException(`Issue not found by ID: ${chalk_1.default.green(id)}`);
                }
                yield this.checkAilment(ailment);
            }
            else {
                const ailments = yield this.detectAilments();
                yield this.checkAilments(ailments);
            }
        });
    }
    checkAilments(ailments) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            let treatableAilments = 0;
            if (ailments.length > 0) {
                for (const ailment of ailments) {
                    if (guards_1.isTreatableAilment(ailment)) {
                        treatableAilments += 1;
                    }
                    yield this.checkAilment(ailment);
                }
            }
            const msg = ('Doctor Summary\n' +
                `- Detected ${chalk_1.default.bold(String(ailments.length))} issue${ailments.length === 1 ? '' : 's'}.` +
                `${ailments.length === 0 ? ' Aww yeah! 💪' : ''}\n` +
                `- ${chalk_1.default.bold(String(treatableAilments))} issue${treatableAilments === 1 ? '' : 's'} can be fixed automatically${treatableAilments > 0 ? ` by running: ${chalk_1.default.green('ionic doctor fix')}` : ''}`);
            if (ailments.length > 0) {
                this.env.log.info(msg);
                throw new errors_1.FatalException(''); // exit 1
            }
            else {
                this.env.log.ok(msg);
            }
        });
    }
    checkAilment(ailment) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { formatAilmentMessage } = yield Promise.resolve().then(() => require('../../lib/doctor'));
            if (yield ailment.detected()) {
                this.env.log.warn(yield formatAilmentMessage(ailment));
            }
            else {
                this.env.log.ok(`${chalk_1.default.green(ailment.id)} was not detected.`);
            }
        });
    }
}
exports.DoctorCheckCommand = DoctorCheckCommand;
