"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const format_1 = require("@ionic/cli-framework/utils/format");
const string_1 = require("@ionic/cli-framework/utils/string");
const chalk_1 = require("chalk");
const guards_1 = require("../../guards");
const base_1 = require("./base");
class DoctorListCommand extends base_1.DoctorCommand {
    getMetadata() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return {
                name: 'list',
                type: 'project',
                summary: 'List all issues and their identifiers',
                description: `
Issues can have various tags:
- ${chalk_1.default.bold('treatable')}: ${chalk_1.default.green('ionic doctor treat')} can attempt to fix the issue
- ${chalk_1.default.bold('ignored')}: configured not to be detected in ${chalk_1.default.green('ionic doctor check')} or ${chalk_1.default.green('ionic doctor treat')}
- ${chalk_1.default.bold('explicit-detection')}: issue is only detected explicitly with ${chalk_1.default.green('ionic doctor check <issue-id>')}

You can flip whether an issue is ignored or not by using ${chalk_1.default.green('ionic config set -g doctor.issues.<issue-id>.ignored true/false')}, where ${chalk_1.default.green('<issue-id>')} matches an ID listed with this command.
      `,
            };
        });
    }
    run(inputs, options) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const registry = yield this.getRegistry();
            const rows = registry.ailments.map(ailment => {
                const tags = [];
                const ignored = this.env.config.get(`doctor.issues.${ailment.id}.ignored`);
                if (ignored) {
                    tags.push('ignored');
                }
                if (guards_1.isTreatableAilment(ailment)) {
                    tags.push('treatable');
                }
                if (!ailment.implicit) {
                    tags.push('explicit-detection');
                }
                return [
                    chalk_1.default.green(ailment.id),
                    ailment.projects ? ailment.projects.map(t => chalk_1.default.bold(t)).join(', ') : 'all',
                    tags.map(t => chalk_1.default.bold(t)).join(', '),
                ];
            });
            rows.sort((row1, row2) => string_1.strcmp(row1[0], row2[0]));
            this.env.log.rawmsg(format_1.columnar(rows, { headers: ['id', 'affected projects', 'tags'] }));
        });
    }
}
exports.DoctorListCommand = DoctorListCommand;
