"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const cli_framework_1 = require("@ionic/cli-framework");
const chalk_1 = require("chalk");
const guards_1 = require("../../guards");
const errors_1 = require("../../lib/errors");
const base_1 = require("./base");
const ERROR_AILMENT_IGNORED = 'AILMENT_IGNORED';
const ERROR_AILMENT_SKIPPED = 'AILMENT_SKIPPED';
class DoctorTreatCommand extends base_1.DoctorCommand {
    getMetadata() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return {
                name: 'treat',
                type: 'project',
                summary: 'Attempt to fix issues in your Ionic project',
                description: `
This command detects and attempts to fix common issues. Before a fix is attempted, the steps are printed and a confirmation prompt is displayed.

Optionally supply the ${chalk_1.default.green('id')} argument to attempt to fix a single issue. Use ${chalk_1.default.green('ionic doctor list')} to list all known issues.
      `,
                exampleCommands: [
                    '',
                    'git-not-used',
                ],
                inputs: [
                    {
                        name: 'id',
                        summary: 'The issue identifier',
                    },
                ],
            };
        });
    }
    run(inputs, options) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { formatAilmentMessage } = yield Promise.resolve().then(() => require('../../lib/doctor'));
            const [id] = inputs;
            if (id) {
                const registry = yield this.getRegistry();
                const ailmentIds = registry.ailments.map(a => a.id);
                cli_framework_1.validate(id, 'id', [cli_framework_1.contains(ailmentIds, {})]);
                const ailment = registry.get(id);
                if (!ailment) {
                    throw new errors_1.FatalException(`Issue not found by ID: ${chalk_1.default.green(id)}`);
                }
                const detected = yield ailment.detected();
                if (!detected) {
                    this.env.log.ok(`${chalk_1.default.green(ailment.id)} does not need fixing as it was not detected.`);
                    return;
                }
                if (!guards_1.isTreatableAilment(ailment)) {
                    this.env.log.warn(yield formatAilmentMessage(ailment));
                    throw new errors_1.FatalException(`Issue cannot be fixed automatically: ${chalk_1.default.green(ailment.id)}\n` +
                        `Unfortunately the CLI can't automatically fix the specified issue at this time. However, please see the steps above for manually fixing the issue.`);
                }
                if (this.env.config.get(`doctor.issues.${ailment.id}.ignored`)) {
                    const confirm = yield this.env.prompt({
                        type: 'confirm',
                        name: 'confirm',
                        message: `${chalk_1.default.green(ailment.id)} is ignored, are you sure you want to continue?`,
                    });
                    if (!confirm) {
                        return;
                    }
                    this.env.config.unset(`doctor.issues.${ailment.id}.ignored`);
                }
                try {
                    yield this.treatAilment(ailment);
                }
                catch (e) {
                    this.handleError(e);
                }
            }
            else {
                const ailments = yield this.detectTreatableAilments();
                yield this.treatAilments(ailments);
            }
        });
    }
    treatAilments(ailments) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            let treatedAilments = 0;
            for (const ailment of ailments) {
                try {
                    const treated = yield this.treatAilment(ailment);
                    if (treated) {
                        treatedAilments += 1;
                    }
                }
                catch (e) {
                    this.handleError(e);
                }
            }
            this.env.log.info('Doctor Summary\n' +
                `- Detected ${chalk_1.default.bold(String(ailments.length))} treatable issue${ailments.length === 1 ? '' : 's'}\n` +
                (treatedAilments > 0 ? `- ${chalk_1.default.bold(String(treatedAilments))} ${treatedAilments === 1 ? 'issue was' : 'issues were'} fixed automatically` : ''));
        });
    }
    handleError(e) {
        if (e !== ERROR_AILMENT_SKIPPED && e !== ERROR_AILMENT_IGNORED) {
            if (guards_1.isExitCodeException(e)) {
                this.env.log.error(`Error occurred during automatic fix: ${e.message}`);
            }
            else {
                this.env.log.error(`Error occurred during automatic fix: ${e.stack ? e.stack : e}`);
            }
        }
    }
    treatAilment(ailment) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { formatAilmentMessage } = yield Promise.resolve().then(() => require('../../lib/doctor'));
            const treatmentSteps = yield ailment.getTreatmentSteps();
            this.env.log.warn(yield formatAilmentMessage(ailment));
            const CHOICE_YES = 'yes';
            const CHOICE_NO = 'no';
            const CHOICE_IGNORE = 'ignore';
            const choice = yield this.env.prompt({
                type: 'list',
                name: 'choice',
                message: `Fix automatically?`,
                choices: [
                    {
                        name: 'Yes',
                        value: CHOICE_YES,
                    },
                    {
                        name: 'No',
                        value: CHOICE_NO,
                    },
                    {
                        name: 'Ignore forever',
                        value: CHOICE_IGNORE,
                    },
                ],
            });
            if (choice === CHOICE_YES) {
                for (const i in treatmentSteps) {
                    const step = treatmentSteps[i];
                    try {
                        yield step.treat();
                    }
                    catch (e) {
                        if (!guards_1.isExitCodeException(e) || e.exitCode > 0) {
                            throw e;
                        }
                    }
                }
                return true;
            }
            else if (choice === CHOICE_NO) {
                throw ERROR_AILMENT_SKIPPED;
            }
            else if (choice === CHOICE_IGNORE) {
                this.env.config.set(`doctor.issues.${ailment.id}.ignored`, true);
                throw ERROR_AILMENT_IGNORED;
            }
            return false;
        });
    }
}
exports.DoctorTreatCommand = DoctorTreatCommand;
