"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const chalk_1 = require("chalk");
const command_1 = require("../../lib/command");
const errors_1 = require("../../lib/errors");
class GitRemoteCommand extends command_1.Command {
    getMetadata() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const dashUrl = this.env.config.getDashUrl();
            return {
                name: 'remote',
                type: 'project',
                summary: 'Adds/updates the Ionic Appflow git remote to your local Ionic app',
                description: `
This command is used by ${chalk_1.default.green('ionic link')} when Ionic Appflow is used as the git host.

${chalk_1.default.green('ionic git remote')} will check the local repository for whether or not the git remote is properly set up. This command operates on the ${chalk_1.default.bold('ionic')} remote. For advanced configuration, see ${chalk_1.default.bold('Settings')} => ${chalk_1.default.bold('Git')} in the app settings of the Dashboard[^dashboard].
      `,
                footnotes: [
                    {
                        id: 'dashboard',
                        url: dashUrl,
                    },
                ],
            };
        });
    }
    run(inputs, options) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { AppClient } = yield Promise.resolve().then(() => require('../../lib/app'));
            const { addIonicRemote, getIonicRemote, initializeRepo, isRepoInitialized, setIonicRemote } = yield Promise.resolve().then(() => require('../../lib/git'));
            if (!this.project) {
                throw new errors_1.FatalException(`Cannot run ${chalk_1.default.green('ionic git remote')} outside a project directory.`);
            }
            const token = this.env.session.getUserToken();
            const id = yield this.project.requireAppflowId();
            const appClient = new AppClient(token, this.env);
            const app = yield appClient.load(id);
            if (!app.repo_url) {
                throw new errors_1.FatalException(`Missing ${chalk_1.default.bold('repo_url')} property in app.`);
            }
            if (!(yield isRepoInitialized(this.project.directory))) {
                yield initializeRepo({ shell: this.env.shell }, this.project.directory);
                this.env.log.warn(`Initializing a git repository for your project.\n` +
                    `Before your first ${chalk_1.default.green('git push ionic master')}, you'll want to commit all the files in your project:\n\n` +
                    `${chalk_1.default.green('git commit -a -m "Initial commit"')}\n`);
            }
            const remote = app.repo_url;
            const found = yield getIonicRemote({ shell: this.env.shell }, this.project.directory);
            if (found) {
                if (remote === found) {
                    this.env.log.msg(`Existing remote ${chalk_1.default.bold('ionic')} found.`);
                }
                else {
                    yield setIonicRemote({ shell: this.env.shell }, this.project.directory, remote);
                    this.env.log.ok(`Updated remote ${chalk_1.default.bold('ionic')}.`);
                }
            }
            else {
                yield addIonicRemote({ shell: this.env.shell }, this.project.directory, remote);
                this.env.log.ok(`Added remote ${chalk_1.default.bold('ionic')}.`);
            }
        });
    }
}
exports.GitRemoteCommand = GitRemoteCommand;
