"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const format_1 = require("@ionic/cli-framework/utils/format");
const string_1 = require("@ionic/cli-framework/utils/string");
const chalk_1 = require("chalk");
const lodash = require("lodash");
const command_1 = require("../lib/command");
const INFO_GROUPS = ['ionic', 'capacitor', 'cordova', 'system', 'environment'];
class InfoCommand extends command_1.Command {
    getMetadata() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return {
                name: 'info',
                type: 'global',
                summary: 'Print project, system, and environment information',
                description: `
This command is an easy way to share information about your setup. If applicable, be sure to run ${chalk_1.default.green('ionic info')} within your project directory to display even more information.
      `,
                options: [
                    {
                        name: 'json',
                        summary: 'Print system/environment info in JSON format',
                        type: Boolean,
                    },
                ],
            };
        });
    }
    run(inputs, options) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { json } = options;
            if (json) {
                process.stdout.write(JSON.stringify(yield this.env.getInfo()));
            }
            else {
                const results = yield this.env.getInfo();
                const groupedInfo = new Map(INFO_GROUPS.map((group) => [group, results.filter(item => item.group === group)]));
                const sortInfo = (a, b) => {
                    if (a.key[0] === '@' && b.key[0] !== '@') {
                        return 1;
                    }
                    if (a.key[0] !== '@' && b.key[0] === '@') {
                        return -1;
                    }
                    return string_1.strcmp(a.key.toLowerCase(), b.key.toLowerCase());
                };
                const projectPath = this.project && this.project.directory;
                const splitInfo = (ary) => ary
                    .sort(sortInfo)
                    .map((item) => [`   ${item.key}${item.flair ? ' ' + chalk_1.default.dim('(' + item.flair + ')') : ''}`, chalk_1.default.dim(item.value) + (item.path && projectPath && !item.path.startsWith(projectPath) ? ` ${chalk_1.default.dim('(' + item.path + ')')}` : '')]);
                const format = (details) => format_1.columnar(details, { vsep: ':' });
                if (!projectPath) {
                    this.env.log.warn('You are not in an Ionic project directory. Project context may be missing.');
                }
                this.env.log.nl();
                for (const [group, info] of groupedInfo.entries()) {
                    if (info.length > 0) {
                        this.env.log.rawmsg(`${chalk_1.default.bold(`${lodash.startCase(group)}:`)}\n\n`);
                        this.env.log.rawmsg(`${format(splitInfo(info))}\n\n`);
                    }
                }
            }
        });
    }
}
exports.InfoCommand = InfoCommand;
