"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const cli_framework_1 = require("@ionic/cli-framework");
const format_1 = require("@ionic/cli-framework/utils/format");
const string_1 = require("@ionic/cli-framework/utils/string");
const chalk_1 = require("chalk");
const path = require("path");
const constants_1 = require("../constants");
const command_1 = require("../lib/command");
const errors_1 = require("../lib/errors");
const project_1 = require("../lib/project");
class InitCommand extends command_1.Command {
    getMetadata() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return {
                name: 'init',
                type: 'global',
                summary: 'Initialize existing projects with Ionic',
                description: `
This command will initialize the current directory with an ${chalk_1.default.bold(constants_1.PROJECT_FILE)} file.

${chalk_1.default.green('ionic init')} will prompt for a project name and then proceed to determine the type of your project. You can specify the ${chalk_1.default.green('name')} argument and ${chalk_1.default.green('--type')} option to provide these values via command-line.
      `,
                exampleCommands: [
                    '',
                    '"My App"',
                    '"My App" --type=angular',
                ],
                inputs: [
                    {
                        name: 'name',
                        summary: `The name of your project (e.g. ${chalk_1.default.green('myApp')}, ${chalk_1.default.green('"My App"')})`,
                        validators: [cli_framework_1.validators.required],
                    },
                ],
                options: [
                    {
                        name: 'type',
                        summary: `Type of project (e.g. ${constants_1.PROJECT_TYPES.map(type => chalk_1.default.green(type)).join(', ')})`,
                    },
                    {
                        name: 'force',
                        summary: 'Initialize even if a project already exists',
                        type: Boolean,
                        aliases: ['f'],
                        default: false,
                    },
                    {
                        name: 'project-id',
                        summary: 'Specify a slug for your app',
                        groups: [cli_framework_1.OptionGroup.Advanced],
                        spec: { value: 'slug' },
                    },
                ],
                groups: [cli_framework_1.CommandGroup.Beta],
            };
        });
    }
    preRun(inputs, options) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const force = options['force'] ? true : false;
            if (this.project && this.project.details.context === 'app' && !force) {
                // TODO: check for existing project config in multi-app
                throw new errors_1.FatalException(`Existing Ionic project file found: ${chalk_1.default.bold(format_1.prettyPath(this.project.filePath))}\n` +
                    `You can re-initialize your project using the ${chalk_1.default.green('--force')} option.`);
            }
            if (!inputs[0]) {
                const name = yield this.env.prompt({
                    type: 'input',
                    name: 'name',
                    message: 'Project name:',
                    validate: v => cli_framework_1.validators.required(v),
                });
                inputs[0] = name;
            }
            if (!options['type']) {
                const details = new project_1.ProjectDetails({ rootDirectory: this.env.ctx.execPath, e: this.env });
                options['type'] = yield details.getTypeFromDetection();
            }
            if (!options['type']) {
                if (this.env.flags.interactive) {
                    this.env.log.warn(`Could not determine project type.\n` +
                        `Please choose a project type from the list.`);
                    this.env.log.nl();
                }
                const type = yield this.env.prompt({
                    type: 'list',
                    name: 'type',
                    message: 'Project type:',
                    choices: constants_1.PROJECT_TYPES.map(t => ({
                        name: `${project_1.prettyProjectName(t)} (${chalk_1.default.green(t)})`,
                        value: t,
                    })),
                });
                options['type'] = type;
            }
        });
    }
    run(inputs, options) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const name = inputs[0].trim();
            const type = options['type'] ? String(options['type']) : undefined;
            const projectId = options['project-id'] ? String(options['project-id']) : string_1.slugify(name); // TODO validate --project-id
            if (!type) {
                throw new errors_1.FatalException(`Could not determine project type.\n` +
                    `Please specify ${chalk_1.default.green('--type')}. See ${chalk_1.default.green('ionic init --help')} for details.`);
            }
            let project;
            if (this.project && this.project.details.context === 'multiapp') {
                project = yield project_1.createProjectFromDetails({ context: 'multiapp', configPath: path.resolve(this.project.rootDirectory, constants_1.PROJECT_FILE), id: projectId, type, errors: [] }, this.env);
                project.config.set('root', path.relative(this.project.rootDirectory, this.env.ctx.execPath));
            }
            else {
                project = yield project_1.createProjectFromDetails({ context: 'app', configPath: path.resolve(this.env.ctx.execPath, constants_1.PROJECT_FILE), type, errors: [] }, this.env);
            }
            project.config.set('name', name);
            project.config.set('type', type);
            this.env.log.ok('Your Ionic project has been initialized!');
        });
    }
}
exports.InitCommand = InitCommand;
