"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const cli_framework_1 = require("@ionic/cli-framework");
const chalk_1 = require("chalk");
const guards_1 = require("../../guards");
const command_1 = require("../../lib/command");
const errors_1 = require("../../lib/errors");
const integrations_1 = require("../../lib/integrations");
class IntegrationsDisableCommand extends command_1.Command {
    getMetadata() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return {
                name: 'disable',
                type: 'project',
                summary: 'Disable an integration',
                description: `
Integrations, such as Cordova, can be disabled with this command.
      `,
                inputs: [
                    {
                        name: 'name',
                        summary: `The integration to disable (e.g. ${integrations_1.INTEGRATION_NAMES.map(i => chalk_1.default.green(i)).join(', ')})`,
                        validators: [cli_framework_1.validators.required, cli_framework_1.contains(integrations_1.INTEGRATION_NAMES, {})],
                    },
                ],
            };
        });
    }
    run(inputs, options) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const [name] = inputs;
            if (!this.project) {
                throw new errors_1.FatalException(`Cannot run ${chalk_1.default.green('ionic integrations disable')} outside a project directory.`);
            }
            if (!guards_1.isIntegrationName(name)) {
                throw new errors_1.FatalException(`Don't know about ${chalk_1.default.green(name)} integration!`);
            }
            const integration = yield this.project.createIntegration(name);
            const integrationsConfig = this.project.config.get('integrations');
            const integrationConfig = integrationsConfig[name];
            try {
                if (!integrationConfig || integrationConfig.enabled === false) {
                    this.env.log.info(`Integration ${chalk_1.default.green(name)} already disabled.`);
                }
                else {
                    yield integration.disable();
                    integrationConfig.enabled = false;
                    this.env.log.ok(`Integration ${chalk_1.default.green(name)} disabled!`);
                }
            }
            catch (e) {
                if (e instanceof cli_framework_1.BaseError) {
                    throw new errors_1.FatalException(e.message);
                }
                throw e;
            }
            this.project.config.set('integrations', integrationsConfig);
        });
    }
}
exports.IntegrationsDisableCommand = IntegrationsDisableCommand;
