"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const cli_framework_1 = require("@ionic/cli-framework");
const chalk_1 = require("chalk");
const path = require("path");
const guards_1 = require("../../guards");
const command_1 = require("../../lib/command");
const errors_1 = require("../../lib/errors");
const integrations_1 = require("../../lib/integrations");
class IntegrationsEnableCommand extends command_1.Command {
    getMetadata() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return {
                name: 'enable',
                type: 'project',
                summary: 'Add & enable integrations to your app',
                description: `
Integrations, such as Cordova, can be enabled with this command. If the integration has never been added to the project, ${chalk_1.default.green('ionic integrations enable')} will download and add the integration.

Integrations can be re-added with the ${chalk_1.default.green('--add')} option.
      `,
                inputs: [
                    {
                        name: 'name',
                        summary: `The integration to enable (e.g. ${integrations_1.INTEGRATION_NAMES.map(i => chalk_1.default.green(i)).join(', ')})`,
                        validators: [cli_framework_1.validators.required, cli_framework_1.contains(integrations_1.INTEGRATION_NAMES, {})],
                    },
                ],
                options: [
                    {
                        name: 'add',
                        summary: 'Download and add the integration even if enabled',
                        type: Boolean,
                    },
                    {
                        name: 'root',
                        summary: 'Specify an alternative destination to download into when adding',
                        spec: { value: 'path' },
                    },
                    {
                        name: 'quiet',
                        summary: 'Do not log file operations',
                        type: Boolean,
                    },
                ],
            };
        });
    }
    run(inputs, options) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const [name] = inputs;
            const { add, quiet } = options;
            if (!this.project) {
                throw new errors_1.FatalException(`Cannot run ${chalk_1.default.green('ionic integrations enable')} outside a project directory.`);
            }
            const root = options['root'] ? path.resolve(this.project.directory, String(options['root'])) : this.project.directory;
            if (!guards_1.isIntegrationName(name)) {
                throw new errors_1.FatalException(`Don't know about ${chalk_1.default.green(name)} integration!`);
            }
            const integration = yield this.project.createIntegration(name);
            const integrationsConfig = this.project.config.get('integrations');
            const integrationConfig = integrationsConfig[name];
            try {
                if (!integrationConfig || add) {
                    yield integration.add({
                        root,
                        enableArgs: options['--'] ? options['--'] : undefined,
                    }, {
                        conflictHandler: (f, stats) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                            const isDirectory = stats.isDirectory();
                            const filename = `${path.basename(f)}${isDirectory ? '/' : ''}`;
                            const type = isDirectory ? 'directory' : 'file';
                            const confirm = yield this.env.prompt({
                                type: 'confirm',
                                name: 'confirm',
                                message: `The ${chalk_1.default.cyan(filename)} ${type} exists in project. Overwrite?`,
                                default: false,
                            });
                            return confirm;
                        }),
                        onFileCreate: f => {
                            if (!quiet) {
                                this.env.log.msg(`${chalk_1.default.green('CREATE')} ${f}`);
                            }
                        },
                    });
                    integrationsConfig[name] = {
                        root: root !== this.project.directory ? path.relative(this.project.rootDirectory, root) : undefined,
                    };
                    this.env.log.ok(`Integration ${chalk_1.default.green(integration.name)} added!`);
                }
                else {
                    const wasEnabled = integrationConfig.enabled !== false;
                    // We still need to run this whenever this command is run to make sure
                    // everything is good with the integration.
                    yield integration.enable();
                    delete integrationConfig.enabled;
                    if (wasEnabled) {
                        this.env.log.info(`Integration ${chalk_1.default.green(integration.name)} enabled.`);
                    }
                    else {
                        this.env.log.ok(`Integration ${chalk_1.default.green(integration.name)} enabled!`);
                    }
                }
            }
            catch (e) {
                if (e instanceof cli_framework_1.BaseError) {
                    throw new errors_1.FatalException(e.message);
                }
                throw e;
            }
            this.project.config.set('integrations', integrationsConfig);
        });
    }
}
exports.IntegrationsEnableCommand = IntegrationsEnableCommand;
