"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const cli_framework_1 = require("@ionic/cli-framework");
const format_1 = require("@ionic/cli-framework/utils/format");
const chalk_1 = require("chalk");
const Debug = require("debug");
const constants_1 = require("../constants");
const guards_1 = require("../guards");
const command_1 = require("../lib/command");
const errors_1 = require("../lib/errors");
const executor_1 = require("../lib/executor");
const open_1 = require("../lib/open");
const debug = Debug('ionic:commands:link');
const CHOICE_CREATE_NEW_APP = 'createNewApp';
const CHOICE_NEVERMIND = 'nevermind';
const CHOICE_RELINK = 'relink';
const CHOICE_LINK_EXISTING_APP = 'linkExistingApp';
const CHOICE_IONIC = 'ionic';
const CHOICE_GITHUB = 'github';
const CHOICE_MASTER_ONLY = 'master';
const CHOICE_SPECIFIC_BRANCHES = 'specific';
class LinkCommand extends command_1.Command {
    getMetadata() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const projectFile = this.project ? format_1.prettyPath(this.project.filePath) : constants_1.PROJECT_FILE;
            return {
                name: 'link',
                type: 'project',
                summary: 'Connect local apps to Ionic Appflow',
                description: `
Link apps on Ionic Appflow to local Ionic projects with this command.

If the ${chalk_1.default.green('id')} argument is excluded, this command will prompt you to select an app from Ionic Appflow.

Ionic Appflow uses a git-based workflow to manage app updates. During the linking process, select ${chalk_1.default.bold('GitHub')} (recommended) or ${chalk_1.default.bold('Ionic Appflow')} as a git host. See our documentation[^appflow-git-basics] for more information.

Ultimately, this command sets the ${chalk_1.default.bold('id')} property in ${chalk_1.default.bold(format_1.prettyPath(projectFile))}, which marks this app as linked.

If you are having issues linking, please get in touch with our Support[^support-request].
      `,
                footnotes: [
                    {
                        id: 'appflow-git-basics',
                        url: 'https://ionicframework.com/docs/appflow/basics/git',
                        shortUrl: 'https://ion.link/appflow-git-basics',
                    },
                    {
                        id: 'support-request',
                        url: 'https://ion.link/support-request',
                    },
                ],
                exampleCommands: ['', 'a1b2c3d4'],
                inputs: [
                    {
                        name: 'id',
                        summary: `The Ionic Appflow ID of the app to link (e.g. ${chalk_1.default.green('a1b2c3d4')})`,
                    },
                ],
                options: [
                    {
                        name: 'name',
                        summary: 'The app name to use during the linking of a new app',
                        groups: [cli_framework_1.OptionGroup.Hidden],
                    },
                    {
                        name: 'create',
                        summary: 'Create a new app on Ionic Appflow and link it with this local Ionic project',
                        type: Boolean,
                        groups: [cli_framework_1.OptionGroup.Hidden],
                    },
                    {
                        name: 'pro-id',
                        summary: 'Specify an app ID from the Ionic Appflow to link',
                        groups: [cli_framework_1.OptionGroup.Deprecated, cli_framework_1.OptionGroup.Hidden],
                        spec: { value: 'id' },
                    },
                ],
            };
        });
    }
    preRun(inputs, options) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { create } = options;
            if (inputs[0] && create) {
                throw new errors_1.FatalException(`Sorry--cannot use both ${chalk_1.default.green('id')} and ${chalk_1.default.green('--create')}. You must either link an existing app or create a new one.`);
            }
            const id = options['pro-id'] ? String(options['pro-id']) : undefined;
            if (id) {
                inputs[0] = id;
            }
        });
    }
    run(inputs, options, runinfo) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { promptToLogin } = yield Promise.resolve().then(() => require('../lib/session'));
            if (!this.project) {
                throw new errors_1.FatalException(`Cannot run ${chalk_1.default.green('ionic link')} outside a project directory.`);
            }
            let id = inputs[0];
            let { create } = options;
            const idFromConfig = this.project.config.get('id');
            if (idFromConfig) {
                if (id && idFromConfig === id) {
                    this.env.log.msg(`Already linked with app ${chalk_1.default.green(id)}.`);
                    return;
                }
                const msg = id ?
                    `Are you sure you want to link it to ${chalk_1.default.green(id)} instead?` :
                    `Would you like to run link again?`;
                const confirm = yield this.env.prompt({
                    type: 'confirm',
                    name: 'confirm',
                    message: `Appflow ID ${chalk_1.default.green(idFromConfig)} is already set up with this app. ${msg}`,
                });
                if (!confirm) {
                    this.env.log.msg('Not linking.');
                    return;
                }
            }
            if (!this.env.session.isLoggedIn()) {
                yield promptToLogin(this.env);
            }
            if (!id && !create) {
                const choices = [
                    {
                        name: `Link ${idFromConfig ? 'a different' : 'an existing'} app on Ionic Appflow`,
                        value: CHOICE_LINK_EXISTING_APP,
                    },
                    {
                        name: 'Create a new app on Ionic Appflow',
                        value: CHOICE_CREATE_NEW_APP,
                    },
                ];
                if (idFromConfig) {
                    choices.unshift({
                        name: `Relink ${chalk_1.default.green(idFromConfig)}`,
                        value: CHOICE_RELINK,
                    });
                }
                const result = yield this.env.prompt({
                    type: 'list',
                    name: 'whatToDo',
                    message: 'What would you like to do?',
                    choices,
                });
                if (result === CHOICE_CREATE_NEW_APP) {
                    create = true;
                    id = undefined;
                }
                else if (result === CHOICE_LINK_EXISTING_APP) {
                    const tasks = this.createTaskChain();
                    tasks.next(`Looking up your apps`);
                    const apps = [];
                    const appClient = yield this.getAppClient();
                    const paginator = appClient.paginate();
                    for (const r of paginator) {
                        const res = yield r;
                        apps.push(...res.data);
                    }
                    tasks.end();
                    if (apps.length === 0) {
                        const confirm = yield this.env.prompt({
                            type: 'confirm',
                            name: 'confirm',
                            message: `No apps found. Would you like to create a new app on Ionic Appflow?`,
                        });
                        if (!confirm) {
                            throw new errors_1.FatalException(`Cannot link without an app selected.`);
                        }
                        create = true;
                        id = undefined;
                    }
                    else {
                        const choice = yield this.chooseApp(apps);
                        if (choice === CHOICE_NEVERMIND) {
                            this.env.log.info('Not linking app.');
                            id = undefined;
                        }
                        else {
                            id = choice;
                        }
                    }
                }
                else if (result === CHOICE_RELINK) {
                    id = idFromConfig;
                }
            }
            if (create) {
                let name = options['name'] ? String(options['name']) : undefined;
                if (!name) {
                    name = yield this.env.prompt({
                        type: 'input',
                        name: 'name',
                        message: 'Please enter a name for your new app:',
                        validate: v => cli_framework_1.validators.required(v),
                    });
                }
                id = yield this.createApp({ name }, runinfo);
            }
            else if (id) {
                const app = yield this.lookUpApp(id);
                yield this.linkApp(app, runinfo);
            }
        });
    }
    getAppClient() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { AppClient } = yield Promise.resolve().then(() => require('../lib/app'));
            const token = this.env.session.getUserToken();
            return new AppClient(token, this.env);
        });
    }
    getUserClient() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { UserClient } = yield Promise.resolve().then(() => require('../lib/user'));
            const token = this.env.session.getUserToken();
            return new UserClient(token, this.env);
        });
    }
    lookUpApp(id) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const tasks = this.createTaskChain();
            tasks.next(`Looking up app ${chalk_1.default.green(id)}`);
            const appClient = yield this.getAppClient();
            const app = yield appClient.load(id); // Make sure the user has access to the app
            tasks.end();
            return app;
        });
    }
    createApp({ name }, runinfo) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const appClient = yield this.getAppClient();
            const org_id = this.env.config.get('org.id');
            const app = yield appClient.create({ name, org_id });
            yield this.linkApp(app, runinfo);
            return app.id;
        });
    }
    linkApp(app, runinfo) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            // TODO: load connections
            // TODO: check for git availability before this
            this.env.log.nl();
            this.env.log.info(`Ionic Appflow uses a git-based workflow to manage app updates.\n` +
                `You will be prompted to set up the git host and repository for this new app. See the docs${chalk_1.default.cyan('[1]')} for more information.\n\n` +
                `${chalk_1.default.cyan('[1]')}: ${chalk_1.default.bold('https://ion.link/appflow-git-basics')}`);
            this.env.log.nl();
            const service = yield this.env.prompt({
                type: 'list',
                name: 'gitService',
                message: 'Which git host would you like to use?',
                choices: [
                    {
                        name: 'GitHub',
                        value: CHOICE_GITHUB,
                    },
                    {
                        name: 'Ionic Appflow',
                        value: CHOICE_IONIC,
                    },
                ],
            });
            let githubUrl;
            if (service === CHOICE_IONIC) {
                if (!this.env.config.get('git.setup')) {
                    yield executor_1.runCommand(runinfo, ['ssh', 'setup']);
                }
                yield executor_1.runCommand(runinfo, ['config', 'set', 'id', `"${app.id}"`, '--json']);
                yield executor_1.runCommand(runinfo, ['git', 'remote']);
            }
            else {
                if (service === CHOICE_GITHUB) {
                    githubUrl = yield this.linkGithub(app);
                }
                yield executor_1.runCommand(runinfo, ['config', 'set', 'id', `"${app.id}"`, '--json']);
            }
            this.env.log.ok(`Project linked with app ${chalk_1.default.green(app.id)}!`);
            if (service === CHOICE_GITHUB) {
                this.env.log.info(`Here are some additional links that can help you with you first push to GitHub:\n` +
                    `${chalk_1.default.bold('Adding GitHub as a remote')}:\n\t${chalk_1.default.bold('https://help.github.com/articles/adding-a-remote/')}\n\n` +
                    `${chalk_1.default.bold('Pushing to a remote')}:\n\t${chalk_1.default.bold('https://help.github.com/articles/pushing-to-a-remote/')}\n\n` +
                    `${chalk_1.default.bold('Working with branches')}:\n\t${chalk_1.default.bold('https://guides.github.com/introduction/flow/')}\n\n` +
                    `${chalk_1.default.bold('More comfortable with a GUI? Try GitHub Desktop!')}\n\t${chalk_1.default.bold('https://desktop.github.com/')}`);
                if (githubUrl) {
                    this.env.log.info(`You can now push to one of your branches on GitHub to trigger a build in Ionic Appflow!\n` +
                        `If you haven't added GitHub as your origin you can do so by running:\n\n` +
                        `${chalk_1.default.green('git remote add origin ' + githubUrl)}\n\n` +
                        `You can find additional links above to help if you're having issues.`);
                }
            }
        });
    }
    linkGithub(app) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { id } = this.env.session.getUser();
            const userClient = yield this.getUserClient();
            const user = yield userClient.load(id, { fields: ['oauth_identities'] });
            if (!user.oauth_identities || !user.oauth_identities.github) {
                yield this.oAuthProcess(id);
            }
            if (yield this.needsAssociation(app, user.id)) {
                yield this.confirmGithubRepoExists();
                const repoId = yield this.selectGithubRepo();
                const branches = yield this.selectGithubBranches(repoId);
                return this.connectGithub(app, repoId, branches);
            }
        });
    }
    confirmGithubRepoExists() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            let confirm = false;
            this.env.log.nl();
            this.env.log.info(chalk_1.default.bold(`In order to link to a GitHub repository the repository must already exist on GitHub.`));
            this.env.log.info(`${chalk_1.default.bold('If the repository does not exist please create one now before continuing.')}\n` +
                `If you're not familiar with Git you can learn how to set it up with GitHub here:\n\n` +
                chalk_1.default.bold(`https://help.github.com/articles/set-up-git/ \n\n`) +
                `You can find documentation on how to create a repository on GitHub and push to it here:\n\n` +
                chalk_1.default.bold(`https://help.github.com/articles/create-a-repo/`));
            confirm = yield this.env.prompt({
                type: 'confirm',
                name: 'confirm',
                message: 'Does the repository exist on GitHub?',
            });
            if (!confirm) {
                throw new errors_1.FatalException(`Repo must exist on GitHub in order to link. Please create the repo and run ${chalk_1.default.green('ionic link')} again.`);
            }
        });
    }
    oAuthProcess(userId) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const userClient = yield this.getUserClient();
            let confirm = false;
            this.env.log.nl();
            this.env.log.info(`GitHub OAuth setup required.\n` +
                `To continue, we need you to authorize Ionic Appflow with your GitHub account. ` +
                `A browser will open and prompt you to complete the authorization request. ` +
                `When finished, please return to the CLI to continue linking your app.`);
            confirm = yield this.env.prompt({
                type: 'confirm',
                name: 'ready',
                message: 'Open browser:',
            });
            if (!confirm) {
                throw new errors_1.FatalException(`GitHub OAuth setup is required to link to GitHub repository. Please run ${chalk_1.default.green('ionic link')} again when ready.`);
            }
            const url = yield userClient.oAuthGithubLogin(userId);
            yield open_1.open(url);
            confirm = yield this.env.prompt({
                type: 'confirm',
                name: 'ready',
                message: 'Authorized and ready to continue:',
            });
            if (!confirm) {
                throw new errors_1.FatalException(`GitHub OAuth setup is required to link to GitHub repository. Please run ${chalk_1.default.green('ionic link')} again when ready.`);
            }
        });
    }
    needsAssociation(app, userId) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const appClient = yield this.getAppClient();
            if (app.association && app.association.repository.html_url) {
                this.env.log.msg(`App ${chalk_1.default.green(app.id)} already connected to ${chalk_1.default.bold(app.association.repository.html_url)}`);
                const confirm = yield this.env.prompt({
                    type: 'confirm',
                    name: 'confirm',
                    message: 'Would you like to connect a different repo?',
                });
                if (!confirm) {
                    return false;
                }
                try {
                    // TODO: maybe we can use a PUT instead of DELETE now + POST later?
                    yield appClient.deleteAssociation(app.id);
                }
                catch (e) {
                    if (guards_1.isSuperAgentError(e)) {
                        if (e.response.status === 401) {
                            yield this.oAuthProcess(userId);
                            yield appClient.deleteAssociation(app.id);
                            return true;
                        }
                        else if (e.response.status === 404) {
                            debug(`DELETE ${app.id} GitHub association not found`);
                            return true;
                        }
                    }
                    throw e;
                }
            }
            return true;
        });
    }
    connectGithub(app, repoId, branches) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const appClient = yield this.getAppClient();
            try {
                const association = yield appClient.createAssociation(app.id, { repoId, type: 'github', branches });
                this.env.log.ok(`App ${chalk_1.default.green(app.id)} connected to ${chalk_1.default.bold(association.repository.html_url)}`);
                return association.repository.html_url;
            }
            catch (e) {
                if (guards_1.isSuperAgentError(e) && e.response.status === 403) {
                    throw new errors_1.FatalException(e.response.body.error.message);
                }
            }
        });
    }
    formatRepoName(fullName) {
        const [org, name] = fullName.split('/');
        return `${chalk_1.default.dim(`${org} /`)} ${name}`;
    }
    chooseApp(apps) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { formatName } = yield Promise.resolve().then(() => require('../lib/app'));
            const neverMindChoice = {
                name: chalk_1.default.bold('Nevermind'),
                id: CHOICE_NEVERMIND,
                value: CHOICE_NEVERMIND,
                org: null,
            };
            const linkedApp = yield this.env.prompt({
                type: 'list',
                name: 'linkedApp',
                message: 'Which app would you like to link',
                choices: [
                    ...apps.map(app => ({
                        name: `${formatName(app)} ${chalk_1.default.dim(`(${app.id})`)}`,
                        value: app.id,
                    })),
                    cli_framework_1.createPromptChoiceSeparator(),
                    neverMindChoice,
                    cli_framework_1.createPromptChoiceSeparator(),
                ],
            });
            return linkedApp;
        });
    }
    selectGithubRepo() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const user = this.env.session.getUser();
            const userClient = yield this.getUserClient();
            const tasks = this.createTaskChain();
            const task = tasks.next('Looking up your GitHub repositories');
            const paginator = userClient.paginateGithubRepositories(user.id);
            const repos = [];
            try {
                for (const r of paginator) {
                    const res = yield r;
                    repos.push(...res.data);
                    task.msg = `Looking up your GitHub repositories: ${chalk_1.default.bold(String(repos.length))} found`;
                }
            }
            catch (e) {
                tasks.fail();
                if (guards_1.isSuperAgentError(e) && e.response.status === 401) {
                    yield this.oAuthProcess(user.id);
                    return this.selectGithubRepo();
                }
                throw e;
            }
            tasks.end();
            const repoId = yield this.env.prompt({
                type: 'list',
                name: 'githubRepo',
                message: 'Which GitHub repository would you like to link?',
                choices: repos.map(repo => ({
                    name: this.formatRepoName(repo.full_name),
                    value: String(repo.id),
                })),
            });
            return Number(repoId);
        });
    }
    selectGithubBranches(repoId) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            this.env.log.nl();
            this.env.log.info(chalk_1.default.bold(`By default Ionic Appflow links only to the ${chalk_1.default.green('master')} branch.`));
            this.env.log.info(`${chalk_1.default.bold('If you\'d like to link to another branch or multiple branches you\'ll need to select each branch to connect to.')}\n` +
                `If you're not familiar with on working with branches in GitHub you can read about them here:\n\n` +
                chalk_1.default.bold(`https://guides.github.com/introduction/flow/ \n\n`));
            const choice = yield this.env.prompt({
                type: 'list',
                name: 'githubMultipleBranches',
                message: 'Which would you like to do?',
                choices: [
                    {
                        name: `Link to master branch only`,
                        value: CHOICE_MASTER_ONLY,
                    },
                    {
                        name: `Link to specific branches`,
                        value: CHOICE_SPECIFIC_BRANCHES,
                    },
                ],
            });
            switch (choice) {
                case CHOICE_MASTER_ONLY:
                    return ['master'];
                case CHOICE_SPECIFIC_BRANCHES:
                    // fall through and begin prompting to choose branches
                    break;
                default:
                    throw new errors_1.FatalException('Aborting. No branch choice specified.');
            }
            const user = this.env.session.getUser();
            const userClient = yield this.getUserClient();
            const paginator = userClient.paginateGithubBranches(user.id, repoId);
            const tasks = this.createTaskChain();
            const task = tasks.next('Looking for available branches');
            const availableBranches = [];
            try {
                for (const r of paginator) {
                    const res = yield r;
                    availableBranches.push(...res.data);
                    task.msg = `Looking up the available branches on your GitHub repository: ${chalk_1.default.bold(String(availableBranches.length))} found`;
                }
            }
            catch (e) {
                tasks.fail();
                throw e;
            }
            tasks.end();
            const choices = availableBranches.map(branch => ({
                name: branch.name,
                value: branch.name,
                checked: branch.name === 'master',
            }));
            if (choices.length === 0) {
                this.env.log.warn(`No branches found for the repository. Linking to ${chalk_1.default.green('master')} branch.`);
                return ['master'];
            }
            const selectedBranches = yield this.env.prompt({
                type: 'checkbox',
                name: 'githubBranches',
                message: 'Which branch would you like to link?',
                choices,
                default: ['master'],
            });
            return selectedBranches;
        });
    }
}
exports.LinkCommand = LinkCommand;
