"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const cli_framework_1 = require("@ionic/cli-framework");
const chalk_1 = require("chalk");
const command_1 = require("../lib/command");
const errors_1 = require("../lib/errors");
const uuid_1 = require("../lib/utils/uuid");
class LoginCommand extends command_1.Command {
    getMetadata() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return {
                name: 'login',
                type: 'global',
                summary: 'Login to Ionic Appflow',
                description: `
Authenticate with Ionic Appflow and retrieve a user token, which is stored in the CLI config.

If the ${chalk_1.default.green('IONIC_TOKEN')} environment variable is set, the CLI will automatically authenticate you. To retrieve your user token, first use ${chalk_1.default.green('ionic login')}, then print the token by running the ${chalk_1.default.green('ionic config get -g tokens.user')} command.

You can also use ${chalk_1.default.green('IONIC_EMAIL')} and ${chalk_1.default.green('IONIC_PASSWORD')} environment variables for automatic authentication, but it is not recommended to store your password in plain text.

If you need to create an Ionic Appflow account, use ${chalk_1.default.green('ionic signup')}.

If you are having issues logging in, please get in touch with our Support[^support-request].
      `,
                footnotes: [
                    {
                        id: 'support-request',
                        url: 'https://ion.link/support-request',
                    },
                ],
                exampleCommands: ['', 'john@example.com', 'hello@example.com secret'],
                inputs: [
                    {
                        name: 'email',
                        summary: 'Your email address',
                        validators: [cli_framework_1.validators.required, cli_framework_1.validators.email],
                        private: true,
                    },
                    {
                        name: 'password',
                        summary: 'Your password',
                        // this is a hack since sso is hidden, no need to make password not required for it
                        validators: process.argv.includes('--sso') ? [] : [cli_framework_1.validators.required],
                        private: true,
                    },
                ],
                options: [
                    {
                        name: 'sso',
                        type: Boolean,
                        summary: 'Open a window to log in with the SSO provider associated with your email',
                        groups: [cli_framework_1.OptionGroup.Hidden],
                    },
                ],
            };
        });
    }
    preRun(inputs, options) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const sso = !!options['sso'];
            if (options['email'] || options['password']) {
                throw new errors_1.FatalException(`${chalk_1.default.green('email')} and ${chalk_1.default.green('password')} are command arguments, not options. Please try this:\n` +
                    `${chalk_1.default.green('ionic login <email> <password>')}\n`);
            }
            const askForEmail = !inputs[0];
            const askForPassword = !sso && !inputs[1];
            if (this.env.session.isLoggedIn()) {
                const email = this.env.config.get('user.email');
                const extra = askForEmail || askForPassword
                    ? (this.env.flags.interactive ? `Prompting for new credentials.\n\nUse ${chalk_1.default.yellow('Ctrl+C')} to cancel and remain logged in.` : '')
                    : 'You will be logged out beforehand.';
                this.env.log.warn('You will be logged out.\n' +
                    `You are already logged in${email ? ' as ' + chalk_1.default.bold(email) : ''}! ${extra}`);
                this.env.log.nl();
            }
            else {
                this.env.log.info(`Log into your Ionic Appflow account!\n` +
                    `If you don't have one yet, create yours by running: ${chalk_1.default.green(`ionic signup`)}`);
                this.env.log.nl();
            }
            // TODO: combine with promptToLogin ?
            if (askForEmail) {
                const email = yield this.env.prompt({
                    type: 'input',
                    name: 'email',
                    message: 'Email:',
                    validate: v => cli_framework_1.validators.required(v) && cli_framework_1.validators.email(v),
                });
                inputs[0] = email;
            }
            if (askForPassword) {
                const password = yield this.env.prompt({
                    type: 'password',
                    name: 'password',
                    message: 'Password:',
                    mask: '*',
                    validate: v => cli_framework_1.validators.required(v),
                });
                inputs[1] = password;
            }
        });
    }
    run(inputs, options) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const [email, password] = inputs;
            const sso = !!options['sso'];
            if (this.env.session.isLoggedIn()) {
                yield this.env.session.logout();
                this.env.config.set('tokens.telemetry', uuid_1.generateUUID());
            }
            if (sso) {
                this.env.log.info(`Ionic Appflow SSO Login\n` +
                    `During this process, a browser window will open to authenticate you with the identity provider for ${chalk_1.default.green(email)}. Please leave this process running until authentication is complete.`);
                this.env.log.nl();
                yield this.env.session.ssoLogin(email);
            }
            else {
                yield this.env.session.login(email, password);
            }
            this.env.log.ok(chalk_1.default.green.bold('You are logged in!'));
        });
    }
}
exports.LoginCommand = LoginCommand;
