"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const format_1 = require("@ionic/cli-framework/utils/format");
const utils_fs_1 = require("@ionic/utils-fs");
const chalk_1 = require("chalk");
const Debug = require("debug");
const path = require("path");
const guards_1 = require("../../guards");
const build_1 = require("../../lib/build");
const command_1 = require("../../lib/command");
const errors_1 = require("../../lib/errors");
const debug = Debug('ionic:commands:monitoring:syncmaps');
const SOURCEMAP_DIRECTORY = '.sourcemaps';
class MonitoringSyncSourcemapsCommand extends command_1.Command {
    getMetadata() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return {
                name: 'syncmaps',
                type: 'project',
                summary: 'Build & upload sourcemaps to Ionic Appflow Monitoring service',
                description: `
By default, ${chalk_1.default.green('ionic monitoring syncmaps')} will upload the sourcemap files within ${chalk_1.default.bold(SOURCEMAP_DIRECTORY)}. To optionally perform a production build before uploading sourcemaps, specify the ${chalk_1.default.green('--build')} flag.
      `,
                inputs: [
                    {
                        name: 'snapshot_id',
                        summary: `Specify a Snapshot ID to associate the uploaded sourcemaps with`,
                    },
                ],
                options: [
                    {
                        name: 'build',
                        summary: 'Invoke a production Ionic build',
                        type: Boolean,
                    },
                ],
            };
        });
    }
    run(inputs, options) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { loadConfigXml } = yield Promise.resolve().then(() => require('../../lib/integrations/cordova/config'));
            if (!this.project) {
                throw new errors_1.FatalException(`Cannot run ${chalk_1.default.green('ionic monitoring syncmaps')} outside a project directory.`);
            }
            const token = this.env.session.getUserToken();
            const appflowId = yield this.project.requireAppflowId();
            const [snapshotId] = inputs;
            const doBuild = options.build ? true : false;
            const cordova = this.project.requireIntegration('cordova');
            const conf = yield loadConfigXml(cordova);
            const cordovaInfo = conf.getProjectInfo();
            const appVersion = cordovaInfo.version;
            const commitHash = (yield this.env.shell.output('git', ['rev-parse', 'HEAD'], { cwd: this.project.directory })).trim();
            debug(`Commit hash: ${chalk_1.default.bold(commitHash)}`);
            const sourcemapsDir = path.resolve(this.project.directory, SOURCEMAP_DIRECTORY);
            let sourcemapsExist = yield utils_fs_1.pathExists(sourcemapsDir);
            if (doBuild || !sourcemapsExist) {
                // TODO: use runner directly
                yield build_1.build({ config: this.env.config, log: this.env.log, shell: this.env.shell, prompt: this.env.prompt, project: this.project }, [], { _: [], prod: true });
            }
            sourcemapsExist = yield utils_fs_1.pathExists(sourcemapsDir);
            if (sourcemapsExist) {
                this.env.log.msg(`Using existing sourcemaps in ${chalk_1.default.bold(format_1.prettyPath(sourcemapsDir))}`);
            }
            else { // TODO: this is hard-coded for ionic-angular, make it work for all project types
                throw new errors_1.FatalException(`Cannot find directory: ${chalk_1.default.bold(format_1.prettyPath(sourcemapsDir))}.\n` +
                    `Make sure you have the latest ${chalk_1.default.bold('@ionic/app-scripts')}. Then, re-run this command.`);
            }
            let count = 0;
            const tasks = this.createTaskChain();
            const syncTask = tasks.next('Syncing sourcemaps');
            const sourcemapFiles = (yield utils_fs_1.readdirSafe(sourcemapsDir)).filter(f => f.endsWith('.js.map'));
            debug(`Found ${sourcemapFiles.length} sourcemap files: ${sourcemapFiles.map(f => chalk_1.default.bold(f)).join(', ')}`);
            yield Promise.all(sourcemapFiles.map((f) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                yield this.syncSourcemap(path.resolve(sourcemapsDir, f), snapshotId, appVersion, commitHash, appflowId, token);
                count += 1;
                syncTask.msg = `Syncing sourcemaps: ${chalk_1.default.bold(`${count} / ${sourcemapFiles.length}`)}`;
            })));
            syncTask.msg = `Syncing sourcemaps: ${chalk_1.default.bold(`${sourcemapFiles.length} / ${sourcemapFiles.length}`)}`;
            tasks.end();
            const details = format_1.columnar([
                ['Appflow ID', chalk_1.default.bold(appflowId)],
                ['Version', chalk_1.default.bold(appVersion)],
                ['Package ID', chalk_1.default.bold(cordovaInfo.id)],
                ['Snapshot ID', snapshotId ? chalk_1.default.bold(snapshotId) : chalk_1.default.dim('not set')],
            ], { vsep: ':' });
            this.env.log.ok(`Sourcemaps synced!\n` +
                details + '\n\n' +
                `See the Error Monitoring docs for usage information and next steps: ${chalk_1.default.bold('https://ionicframework.com/docs/appflow/monitoring')}`);
        });
    }
    syncSourcemap(file, snapshotId, appVersion, commitHash, appflowId, token) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { req } = yield this.env.client.make('POST', `/monitoring/${appflowId}/sourcemaps`);
            req
                .set('Authorization', `Bearer ${token}`)
                .send({
                name: path.basename(file),
                version: appVersion,
                commit: commitHash,
                snapshot_id: snapshotId,
            });
            try {
                const res = yield this.env.client.do(req);
                return this.uploadSourcemap(res, file);
            }
            catch (e) {
                if (guards_1.isSuperAgentError(e)) {
                    this.env.log.error(`Unable to sync map ${file}: ` + e.message);
                    if (e.response.status === 401) {
                        this.env.log.error('Try logging out and back in again.');
                    }
                }
                else {
                    throw e;
                }
            }
        });
    }
    uploadSourcemap(sourcemap, file) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { createRequest } = yield Promise.resolve().then(() => require('../../lib/utils/http'));
            const sm = sourcemap;
            const fileData = yield utils_fs_1.readFile(file, { encoding: 'utf8' });
            const sourcemapPost = sm.data.sourcemap_post;
            const { req } = yield createRequest('POST', sourcemapPost.url, this.env.config.getHTTPConfig());
            req
                .field(sourcemapPost.fields)
                .field('file', fileData);
            const res = yield req;
            if (res.status !== 204) {
                throw new errors_1.FatalException(`Unexpected status code from AWS: ${res.status}`);
            }
        });
    }
}
exports.MonitoringSyncSourcemapsCommand = MonitoringSyncSourcemapsCommand;
