"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const cli_framework_1 = require("@ionic/cli-framework");
const format_1 = require("@ionic/cli-framework/utils/format");
const process_1 = require("@ionic/cli-framework/utils/process");
const utils_fs_1 = require("@ionic/utils-fs");
const chalk_1 = require("chalk");
const Debug = require("debug");
const fs = require("fs");
const guards_1 = require("../../guards");
const command_1 = require("../../lib/command");
const errors_1 = require("../../lib/errors");
const file_1 = require("../../lib/utils/file");
const http_1 = require("../../lib/utils/http");
const debug = Debug('ionic:commands:package:build');
const PLATFORMS = ['android', 'ios'];
const ANDROID_BUILD_TYPES = ['debug', 'release'];
const IOS_BUILD_TYPES = ['development', 'ad-hoc', 'app-store', 'enterprise'];
const BUILD_TYPES = ANDROID_BUILD_TYPES.concat(IOS_BUILD_TYPES);
const TARGET_PLATFORM = ['Android', 'iOS - Xcode 10 (Preferred)', 'iOS - Xcode 9', 'iOS - Xcode 8'];
class BuildCommand extends command_1.Command {
    getMetadata() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const dashUrl = this.env.config.getDashUrl();
            return {
                name: 'build',
                type: 'project',
                summary: 'Create a package build on Appflow',
                description: `
This command creates a package build on Ionic Appflow. While the build is running, it prints the remote build log to the terminal. If the build is successful, it downloads the created app package file in the current directory.

Apart from ${chalk_1.default.green('--commit')}, every option can be specified using the full name setup within the Appflow Dashboard[^dashboard].

The ${chalk_1.default.green('--security-profile')} option is mandatory for any iOS build but not for Android debug builds.

Customizing the build:
- The ${chalk_1.default.green('--environment')} and ${chalk_1.default.green('--native-config')} options can be used to customize the groups of values exposed to the build.
- Override the preferred platform with ${chalk_1.default.green('--target-platform')}. This is useful for building older iOS apps.
`,
                footnotes: [
                    {
                        id: 'dashboard',
                        url: dashUrl,
                    },
                ],
                exampleCommands: [
                    'android debug',
                    'ios development --security-profile="iOS Security Profile Name"',
                    'android debug --environment="My Custom Environment Name"',
                    'android debug --native-config="My Custom Native Config Name"',
                    'android debug --commit=2345cd3305a1cf94de34e93b73a932f25baac77c',
                    'ios development --security-profile="iOS Security Profile Name" --target-platform="iOS - Xcode 9"',
                    'ios development --security-profile="iOS Security Profile Name" --build-file-name=my_custom_file_name.ipa',
                ],
                inputs: [
                    {
                        name: 'platform',
                        summary: `The platform to package (${PLATFORMS.map(v => chalk_1.default.green(v)).join(', ')})`,
                        validators: [cli_framework_1.validators.required, cli_framework_1.contains(PLATFORMS, {})],
                    },
                    {
                        name: 'type',
                        summary: `The build type (${BUILD_TYPES.map(v => chalk_1.default.green(v)).join(', ')})`,
                        validators: [cli_framework_1.validators.required, cli_framework_1.contains(BUILD_TYPES, {})],
                    },
                ],
                options: [
                    {
                        name: 'security-profile',
                        summary: 'Security profile',
                        type: String,
                        spec: { value: 'name' },
                    },
                    {
                        name: 'environment',
                        summary: 'The group of environment variables exposed to your build',
                        type: String,
                        spec: { value: 'name' },
                    },
                    {
                        name: 'native-config',
                        summary: 'The group of native config variables exposed to your build',
                        type: String,
                        spec: { value: 'name' },
                    },
                    {
                        name: 'commit',
                        summary: 'Commit (defaults to HEAD)',
                        type: String,
                        groups: [cli_framework_1.OptionGroup.Advanced],
                        spec: { value: 'sha1' },
                    },
                    {
                        name: 'target-platform',
                        summary: `Target platform (${TARGET_PLATFORM.map(v => chalk_1.default.green(`"${v}"`)).join(', ')})`,
                        type: String,
                        groups: [cli_framework_1.OptionGroup.Advanced],
                        spec: { value: 'name' },
                    },
                    {
                        name: 'build-file-name',
                        summary: 'The name for the downloaded build file',
                        type: String,
                        groups: [cli_framework_1.OptionGroup.Advanced],
                        spec: { value: 'name' },
                    },
                ],
            };
        });
    }
    preRun(inputs, options) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            if (!inputs[0]) {
                const platformInput = yield this.env.prompt({
                    type: 'list',
                    name: 'platform',
                    choices: PLATFORMS,
                    message: `Platform to package:`,
                    validate: v => cli_framework_1.validators.required(v) && cli_framework_1.contains(PLATFORMS, {})(v),
                });
                inputs[0] = platformInput;
            }
            const buildTypes = inputs[0] === 'ios' ? IOS_BUILD_TYPES : ANDROID_BUILD_TYPES;
            // validate that the build type is valid for the platform
            let reenterBuilType = false;
            if (inputs[1] && !buildTypes.includes(inputs[1])) {
                reenterBuilType = true;
                this.env.log.nl();
                this.env.log.warn(`Build type ${chalk_1.default.bold(inputs[1])} incompatible for ${chalk_1.default.bold(inputs[0])}; please choose a correct one`);
            }
            if (!inputs[1] || reenterBuilType) {
                const typeInput = yield this.env.prompt({
                    type: 'list',
                    name: 'type',
                    choices: buildTypes,
                    message: `Build type:`,
                    validate: v => cli_framework_1.validators.required(v) && cli_framework_1.contains(buildTypes, {})(v),
                });
                inputs[1] = typeInput;
            }
            // the security profile is mandatory for iOS packages, so prompting if it is missing
            if (inputs[0] === 'ios' && !options['security-profile']) {
                if (this.env.flags.interactive) {
                    this.env.log.nl();
                    this.env.log.warn(`A security profile is mandatory to build an iOS package`);
                }
                const securityProfileOption = yield this.env.prompt({
                    type: 'input',
                    name: 'security-profile',
                    message: `Security Profile Name:`,
                });
                options['security-profile'] = securityProfileOption;
            }
        });
    }
    run(inputs, options) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            if (!this.project) {
                throw new errors_1.FatalException(`Cannot run ${chalk_1.default.green('ionic package build')} outside a project directory.`);
            }
            const token = this.env.session.getUserToken();
            const appflowId = yield this.project.requireAppflowId();
            const [platform, buildType] = inputs;
            if (!options.commit) {
                options.commit = (yield this.env.shell.output('git', ['rev-parse', 'HEAD'], { cwd: this.project.directory })).trim();
                debug(`Commit hash: ${chalk_1.default.bold(options.commit)}`);
            }
            let build = yield this.createPackageBuild(appflowId, token, platform, buildType, options);
            const buildId = build.job_id;
            let customBuildFileName = '';
            if (options['build-file-name']) {
                if (typeof (options['build-file-name']) !== 'string' || !file_1.fileUtils.isValidFileName(options['build-file-name'])) {
                    throw new errors_1.FatalException(`${chalk_1.default.bold(String(options['build-file-name']))} is not a valid file name`);
                }
                customBuildFileName = String(options['build-file-name']);
            }
            const details = format_1.columnar([
                ['Appflow ID', chalk_1.default.bold(appflowId)],
                ['Build ID', chalk_1.default.bold(buildId.toString())],
                ['Commit', chalk_1.default.bold(`${build.commit.sha.substring(0, 6)} ${build.commit.note}`)],
                ['Target Platform', chalk_1.default.bold(build.stack.friendly_name)],
                ['Build Type', chalk_1.default.bold(build.build_type)],
                ['Security Profile', build.profile_tag ? chalk_1.default.bold(build.profile_tag) : chalk_1.default.dim('not set')],
                ['Environment', build.environment_name ? chalk_1.default.bold(build.environment_name) : chalk_1.default.dim('not set')],
                ['Native Config', build.native_config_name ? chalk_1.default.bold(build.native_config_name) : chalk_1.default.dim('not set')],
            ], { vsep: ':' });
            this.env.log.ok(`Build created\n` +
                details + '\n\n');
            build = yield this.tailBuildLog(appflowId, buildId, token);
            if (build.state !== 'success') {
                throw new Error('Build failed');
            }
            const url = yield this.getDownloadUrl(appflowId, buildId, token);
            if (!url.url) {
                throw new Error('Missing URL in response');
            }
            const filename = yield this.downloadBuild(url.url, customBuildFileName);
            this.env.log.ok(`Build completed: ${filename}`);
        });
    }
    createPackageBuild(appflowId, token, platform, buildType, options) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { req } = yield this.env.client.make('POST', `/apps/${appflowId}/packages/verbose_post`);
            req.set('Authorization', `Bearer ${token}`).send({
                platform,
                build_type: buildType,
                commit_sha: options.commit,
                stack_name: options['target-platform'],
                profile_name: options['security-profile'],
                environment_name: options.environment,
                native_config_name: options['native-config'],
            });
            try {
                const res = yield this.env.client.do(req);
                return res.data;
            }
            catch (e) {
                if (guards_1.isSuperAgentError(e)) {
                    if (e.response.status === 401) {
                        this.env.log.error('Try logging out and back in again.');
                    }
                    const apiErrorMessage = (e.response.body.error && e.response.body.error.message) ? e.response.body.error.message : 'Api Error';
                    throw new errors_1.FatalException(`Unable to create build: ` + apiErrorMessage);
                }
                else {
                    throw e;
                }
            }
        });
    }
    getPackageBuild(appflowId, buildId, token) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { req } = yield this.env.client.make('GET', `/apps/${appflowId}/packages/${buildId}`);
            req.set('Authorization', `Bearer ${token}`).send();
            try {
                const res = yield this.env.client.do(req);
                return res.data;
            }
            catch (e) {
                if (guards_1.isSuperAgentError(e)) {
                    if (e.response.status === 401) {
                        this.env.log.error('Try logging out and back in again.');
                    }
                    const apiErrorMessage = (e.response.body.error && e.response.body.error.message) ? e.response.body.error.message : 'Api Error';
                    throw new errors_1.FatalException(`Unable to get build ${buildId}: ` + apiErrorMessage);
                }
                else {
                    throw e;
                }
            }
        });
    }
    getDownloadUrl(appflowId, buildId, token) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { req } = yield this.env.client.make('GET', `/apps/${appflowId}/packages/${buildId}/download`);
            req.set('Authorization', `Bearer ${token}`).send();
            try {
                const res = yield this.env.client.do(req);
                return res.data;
            }
            catch (e) {
                if (guards_1.isSuperAgentError(e)) {
                    if (e.response.status === 401) {
                        this.env.log.error('Try logging out and back in again.');
                    }
                    const apiErrorMessage = (e.response.body.error && e.response.body.error.message) ? e.response.body.error.message : 'Api Error';
                    throw new errors_1.FatalException(`Unable to get download URL for build ${buildId}: ` + apiErrorMessage);
                }
                else {
                    throw e;
                }
            }
        });
    }
    tailBuildLog(appflowId, buildId, token) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            let build;
            let start = 0;
            const ws = this.env.log.createWriteStream(cli_framework_1.LOGGER_LEVELS.INFO, false);
            let isCreatedMessage = false;
            while (!(build && (build.state === 'success' || build.state === 'failed'))) {
                yield process_1.sleep(5000);
                build = yield this.getPackageBuild(appflowId, buildId, token);
                if (build && build.state === 'created' && !isCreatedMessage) {
                    ws.write(chalk_1.default.yellow('Concurrency limit reached: build will start as soon as other builds finish.'));
                    isCreatedMessage = true;
                }
                const trace = build.job.trace;
                if (trace.length > start) {
                    ws.write(trace.substring(start));
                    start = trace.length;
                }
            }
            ws.end();
            return build;
        });
    }
    downloadBuild(url, filename) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { req } = yield http_1.createRequest('GET', url, this.env.config.getHTTPConfig());
            if (!filename) {
                req.on('response', res => {
                    const contentDisposition = res.headers['content-disposition'];
                    filename = contentDisposition ? contentDisposition.split('=')[1] : 'output.bin';
                });
            }
            const tmpFile = utils_fs_1.tmpfilepath('ionic-package-build');
            const ws = fs.createWriteStream(tmpFile);
            yield http_1.download(req, ws, {});
            fs.renameSync(tmpFile, filename);
            return filename;
        });
    }
}
exports.BuildCommand = BuildCommand;
