"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const format_1 = require("@ionic/cli-framework/utils/format");
const utils_fs_1 = require("@ionic/utils-fs");
const chalk_1 = require("chalk");
const path = require("path");
const command_1 = require("../lib/command");
const errors_1 = require("../lib/errors");
const executor_1 = require("../lib/executor");
class RepairCommand extends command_1.Command {
    getMetadata() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return {
                name: 'repair',
                type: 'project',
                summary: 'Remove and recreate dependencies and generated files',
                description: `
This command may be useful when obscure errors or issues are encountered. It removes and recreates dependencies of your project.

For Cordova apps, it removes and recreates the generated native project and the native dependencies of your project.
`,
            };
        });
    }
    run(inputs, options, runinfo) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            if (!this.project) {
                throw new errors_1.FatalException(`Cannot run ${chalk_1.default.green('ionic repair')} outside a project directory.`);
            }
            const { pkgManagerArgs } = yield Promise.resolve().then(() => require('../lib/utils/npm'));
            const [installer, ...installerArgs] = yield pkgManagerArgs(this.env.config.get('npmClient'), { command: 'install' });
            const cordova = this.project.getIntegration('cordova');
            if (this.env.flags.interactive) {
                this.env.log.info(`${chalk_1.default.green('ionic repair')} will do the following:\n\n` +
                    `- Remove ${chalk_1.default.bold('node_modules/')} and ${chalk_1.default.bold('package-lock.json')}\n` +
                    `- Run ${chalk_1.default.green([installer, ...installerArgs].join(' '))} to restore dependencies\n` +
                    (cordova && cordova.enabled ?
                        `- Remove ${chalk_1.default.bold('platforms/')} and ${chalk_1.default.bold('plugins/')}\n` +
                            `- Run ${chalk_1.default.green('cordova prepare')} to restore platforms and plugins\n` : ''));
            }
            const confirm = yield this.env.prompt({
                type: 'confirm',
                name: 'confirm',
                message: 'Continue?',
                default: false,
            });
            if (!confirm) {
                throw new errors_1.FatalException(`Not running ${chalk_1.default.green('ionic repair')}.`);
            }
            this.env.log.nl();
            yield this.npmRepair(this.project);
            yield this.cordovaRepair(this.project, runinfo);
        });
    }
    npmRepair(project) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { pkgManagerArgs } = yield Promise.resolve().then(() => require('../lib/utils/npm'));
            const [installer, ...installerArgs] = yield pkgManagerArgs(this.env.config.get('npmClient'), { command: 'install' });
            const tasks = this.createTaskChain();
            const packageLockFile = path.resolve(project.directory, 'package-lock.json');
            const nodeModulesDir = path.resolve(project.directory, 'node_modules');
            tasks.next(`Removing ${chalk_1.default.bold(format_1.prettyPath(packageLockFile))}`);
            yield utils_fs_1.unlink(packageLockFile);
            tasks.next(`Removing ${chalk_1.default.bold(format_1.prettyPath(nodeModulesDir))}`);
            yield utils_fs_1.remove(nodeModulesDir);
            tasks.end();
            yield this.env.shell.run(installer, installerArgs, { cwd: project.directory, stdio: 'inherit' });
        });
    }
    cordovaRepair(project, runinfo) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const tasks = this.createTaskChain();
            const cordova = project.getIntegration('cordova');
            if (cordova && cordova.enabled) {
                const platformsDir = path.resolve(cordova.root, 'platforms');
                const pluginsDir = path.resolve(cordova.root, 'plugins');
                tasks.next(`Removing ${chalk_1.default.bold(format_1.prettyPath(platformsDir))}`);
                yield utils_fs_1.remove(platformsDir);
                tasks.next(`Removing ${chalk_1.default.bold(format_1.prettyPath(pluginsDir))}`);
                yield utils_fs_1.remove(pluginsDir);
                tasks.end();
                yield executor_1.runCommand(runinfo, ['cordova', 'prepare', '--no-build']);
            }
        });
    }
}
exports.RepairCommand = RepairCommand;
