"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const cli_framework_1 = require("@ionic/cli-framework");
const process_1 = require("@ionic/cli-framework/utils/process");
const chalk_1 = require("chalk");
const lodash = require("lodash");
const command_1 = require("../lib/command");
const errors_1 = require("../lib/errors");
const serve_1 = require("../lib/serve");
class ServeCommand extends command_1.Command {
    getMetadata() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            let groups = [];
            const options = [
                ...serve_1.COMMON_SERVE_COMMAND_OPTIONS,
                {
                    name: 'lab-host',
                    summary: 'Use specific address for Ionic Lab server',
                    default: 'localhost',
                    groups: [cli_framework_1.OptionGroup.Advanced],
                    spec: { value: 'host' },
                },
                {
                    name: 'lab-port',
                    summary: 'Use specific port for Ionic Lab server',
                    default: serve_1.DEFAULT_LAB_PORT.toString(),
                    groups: [cli_framework_1.OptionGroup.Advanced],
                    spec: { value: 'port' },
                },
                {
                    name: 'open',
                    summary: 'Do not open a browser window',
                    type: Boolean,
                    default: true,
                },
                {
                    name: 'local',
                    summary: 'Disable external network usage',
                    type: Boolean,
                },
                {
                    name: 'browser',
                    summary: `Specifies the browser to use (${serve_1.BROWSERS.map(b => chalk_1.default.green(b)).join(', ')})`,
                    aliases: ['w'],
                    groups: [cli_framework_1.OptionGroup.Advanced],
                },
                {
                    name: 'browseroption',
                    summary: `Specifies a path to open to (${chalk_1.default.green('/#/tab/dash')})`,
                    aliases: ['o'],
                    groups: [cli_framework_1.OptionGroup.Advanced],
                    spec: { value: 'path' },
                },
                {
                    name: 'lab',
                    summary: 'Test your apps on multiple platform types in the browser',
                    type: Boolean,
                    aliases: ['l'],
                },
            ];
            const exampleCommands = ['', '--local', '--lab'];
            const footnotes = [];
            let description = `
Easily spin up a development server which launches in your browser. It watches for changes in your source files and automatically reloads with the updated build.

By default, ${chalk_1.default.green('ionic serve')} boots up a development server on all network interfaces and prints the external address(es) on which your app is being served. It also broadcasts your app to the Ionic DevApp on your network. To disable the DevApp and bind to ${chalk_1.default.green('localhost')}, use ${chalk_1.default.green('--local')}.

Try the ${chalk_1.default.green('--lab')} option to see multiple platforms at once.`;
            const runner = this.project && (yield this.project.getServeRunner());
            if (runner) {
                const libmetadata = yield runner.getCommandMetadata();
                groups = libmetadata.groups || [];
                options.push(...libmetadata.options || []);
                description += `\n\n${(libmetadata.description || '').trim()}`;
                footnotes.push(...libmetadata.footnotes || []);
                exampleCommands.push(...libmetadata.exampleCommands || []);
            }
            return {
                name: 'serve',
                type: 'project',
                summary: 'Start a local dev server for app dev/testing',
                description,
                footnotes,
                groups,
                exampleCommands,
                options,
            };
        });
    }
    preRun(inputs, options, runinfo) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const lastpath = lodash.last(runinfo.location.path);
            const alias = lastpath ? lastpath[0] : undefined;
            if (alias === 'lab') {
                options['lab'] = true;
            }
            if (options['nolivereload']) {
                this.env.log.warn(`The ${chalk_1.default.green('--nolivereload')} option has been deprecated. Please use ${chalk_1.default.green('--no-livereload')}.`);
                options['livereload'] = false;
            }
            if (options['nobrowser']) {
                this.env.log.warn(`The ${chalk_1.default.green('--nobrowser')} option has been deprecated. Please use ${chalk_1.default.green('--no-open')}.`);
                options['open'] = false;
            }
            if (options['b']) {
                options['open'] = false;
            }
            if (options['noproxy']) {
                this.env.log.warn(`The ${chalk_1.default.green('--noproxy')} option has been deprecated. Please use ${chalk_1.default.green('--no-proxy')}.`);
                options['proxy'] = false;
            }
            if (options['x']) {
                options['proxy'] = false;
            }
        });
    }
    run(inputs, options) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            if (!this.project) {
                throw new errors_1.FatalException(`Cannot run ${chalk_1.default.green('ionic serve')} outside a project directory.`);
            }
            // TODO: use runner directly
            yield serve_1.serve({ flags: this.env.flags, config: this.env.config, log: this.env.log, prompt: this.env.prompt, shell: this.env.shell, project: this.project }, inputs, options);
            yield process_1.sleepForever();
        });
    }
}
exports.ServeCommand = ServeCommand;
class LabCommand extends ServeCommand {
    getMetadata() {
        const _super = Object.create(null, {
            getMetadata: { get: () => super.getMetadata }
        });
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const metadata = yield _super.getMetadata.call(this);
            const groups = [...metadata.groups || [], cli_framework_1.CommandGroup.Hidden];
            const exampleCommands = [...metadata.exampleCommands || []].filter(c => !c.includes('--lab'));
            return Object.assign({}, metadata, { summary: 'Start Ionic Lab for multi-platform dev/testing', description: `
Start an instance of ${chalk_1.default.bold('Ionic Lab')}, a tool for developing Ionic apps for multiple platforms at once side-by-side.

${chalk_1.default.green('ionic lab')} is just a convenient shortcut for ${chalk_1.default.green('ionic serve --lab')}.`, groups,
                exampleCommands });
        });
    }
}
exports.LabCommand = LabCommand;
