"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const cli_framework_1 = require("@ionic/cli-framework");
const format_1 = require("@ionic/cli-framework/utils/format");
const utils_fs_1 = require("@ionic/utils-fs");
const chalk_1 = require("chalk");
const fs = require("fs");
const os = require("os");
const path = require("path");
const guards_1 = require("../../guards");
const errors_1 = require("../../lib/errors");
const executor_1 = require("../../lib/executor");
const base_1 = require("./base");
class SSHAddCommand extends base_1.SSHBaseCommand {
    getMetadata() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return {
                name: 'add',
                type: 'global',
                summary: 'Add an SSH public key to Ionic',
                inputs: [
                    {
                        name: 'pubkey-path',
                        summary: 'Location of public key file to add to Ionic',
                        validators: [cli_framework_1.validators.required],
                    },
                ],
                options: [
                    {
                        name: 'use',
                        summary: 'Use the newly added key as your default SSH key for Ionic',
                        type: Boolean,
                    },
                ],
            };
        });
    }
    preRun(inputs, options) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            if (!inputs[0]) {
                const defaultPubkeyPath = path.resolve(os.homedir(), '.ssh', 'id_rsa.pub');
                const defaultPubkeyExists = yield utils_fs_1.pathAccessible(defaultPubkeyPath, fs.constants.R_OK);
                const pubkeyPath = yield this.env.prompt({
                    type: 'input',
                    name: 'pubkeyPath',
                    message: 'Enter the location to your public key file to upload to Ionic:',
                    default: defaultPubkeyExists ? format_1.prettyPath(defaultPubkeyPath) : undefined,
                });
                inputs[0] = pubkeyPath;
            }
        });
    }
    run(inputs, options, runinfo) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { ERROR_SSH_INVALID_PUBKEY, SSHKeyClient, parsePublicKeyFile } = yield Promise.resolve().then(() => require('../../lib/ssh'));
            const pubkeyPath = format_1.expandPath(inputs[0]);
            const pubkeyName = format_1.prettyPath(pubkeyPath);
            let pubkey;
            try {
                [pubkey] = yield parsePublicKeyFile(pubkeyPath);
            }
            catch (e) {
                if (e.code === 'ENOENT') {
                    throw new errors_1.FatalException(`${chalk_1.default.bold(format_1.prettyPath(pubkeyPath))} does not appear to exist. Please specify a valid SSH public key.\n` +
                        `If you are having issues, try using ${chalk_1.default.green('ionic ssh setup')}.`);
                }
                else if (e === ERROR_SSH_INVALID_PUBKEY) {
                    throw new errors_1.FatalException(`${chalk_1.default.bold(pubkeyName)} does not appear to be a valid SSH public key. (Not in ${chalk_1.default.bold('authorized_keys')} file format.)\n` +
                        `If you are having issues, try using ${chalk_1.default.green('ionic ssh setup')}.`);
                }
                throw e;
            }
            const user = this.env.session.getUser();
            const token = this.env.session.getUserToken();
            const sshkeyClient = new SSHKeyClient({ client: this.env.client, token, user });
            try {
                const key = yield sshkeyClient.create({ pubkey });
                this.env.log.ok(`Your public key (${chalk_1.default.bold(key.fingerprint)}) has been added to Ionic!`);
            }
            catch (e) {
                if (guards_1.isSuperAgentError(e) && e.response.status === 409) {
                    this.env.log.msg('Pubkey already added to Ionic.');
                }
                else {
                    throw e;
                }
            }
            if (pubkeyPath.endsWith('.pub')) {
                let confirm = options['use'];
                if (!confirm) {
                    confirm = yield this.env.prompt({
                        type: 'confirm',
                        name: 'confirm',
                        message: 'Would you like to use this key as your default for Ionic?',
                    });
                }
                if (confirm) {
                    const keyPath = pubkeyPath.substring(0, pubkeyPath.length - 4); // corresponding private key, theoretically
                    const keyExists = yield utils_fs_1.pathExists(keyPath);
                    if (keyExists) {
                        yield executor_1.runCommand(runinfo, ['ssh', 'use', format_1.prettyPath(keyPath)]);
                    }
                    else {
                        this.env.log.error(`SSH key does not exist: ${chalk_1.default.bold(format_1.prettyPath(keyPath))}.\n` +
                            `Please use ${chalk_1.default.green('ionic ssh use')} manually to use the corresponding private key.`);
                    }
                }
            }
        });
    }
}
exports.SSHAddCommand = SSHAddCommand;
