"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const cli_framework_1 = require("@ionic/cli-framework");
const chalk_1 = require("chalk");
const base_1 = require("./base");
class SSHDeleteCommand extends base_1.SSHBaseCommand {
    getMetadata() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return {
                name: 'delete',
                type: 'global',
                summary: 'Delete an SSH public key from Ionic',
                inputs: [
                    {
                        name: 'key-id',
                        summary: 'The ID of the public key to delete',
                        validators: [cli_framework_1.validators.required],
                    },
                ],
            };
        });
    }
    preRun(inputs, options) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { SSHKeyClient } = yield Promise.resolve().then(() => require('../../lib/ssh'));
            if (!inputs[0]) {
                const user = this.env.session.getUser();
                const token = this.env.session.getUserToken();
                const sshkeyClient = new SSHKeyClient({ client: this.env.client, user, token });
                const paginator = sshkeyClient.paginate();
                const [r] = paginator;
                const res = yield r;
                if (res.data.length === 0) {
                    this.env.log.warn(`No SSH keys found. Use ${chalk_1.default.green('ionic ssh add')} to add keys to Ionic.`);
                }
                inputs[0] = yield this.env.prompt({
                    type: 'list',
                    name: 'id',
                    message: 'Which SSH keys would you like to delete from Ionic?',
                    choices: res.data.map(key => ({
                        name: `${key.fingerprint} ${key.name} ${key.annotation}`,
                        value: key.id,
                    })),
                });
            }
        });
    }
    run(inputs, options) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { SSHKeyClient } = yield Promise.resolve().then(() => require('../../lib/ssh'));
            const [id] = inputs;
            const user = this.env.session.getUser();
            const token = this.env.session.getUserToken();
            const sshkeyClient = new SSHKeyClient({ client: this.env.client, user, token });
            yield sshkeyClient.delete(id);
            this.env.log.ok(`Your public key (${chalk_1.default.bold(id)}) has been removed from Ionic.`);
        });
    }
}
exports.SSHDeleteCommand = SSHDeleteCommand;
