"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const cli_framework_1 = require("@ionic/cli-framework");
const format_1 = require("@ionic/cli-framework/utils/format");
const utils_fs_1 = require("@ionic/utils-fs");
const chalk_1 = require("chalk");
const path = require("path");
const base_1 = require("./base");
const SSH_KEY_TYPES = ['ecdsa', 'ed25519', 'rsa'];
class SSHGenerateCommand extends base_1.SSHBaseCommand {
    getMetadata() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return {
                name: 'generate',
                type: 'global',
                summary: 'Generates a private and public SSH key pair',
                inputs: [
                    {
                        name: 'key-path',
                        summary: 'Destination of private key file',
                    },
                ],
                options: [
                    {
                        name: 'type',
                        summary: `The type of key to generate: ${SSH_KEY_TYPES.map(v => chalk_1.default.green(v)).join(', ')}`,
                        default: 'rsa',
                        aliases: ['t'],
                        groups: [cli_framework_1.OptionGroup.Advanced],
                    },
                    {
                        name: 'bits',
                        summary: 'Number of bits in the key',
                        aliases: ['b'],
                        default: '2048',
                        groups: [cli_framework_1.OptionGroup.Advanced],
                    },
                    {
                        name: 'annotation',
                        summary: 'Annotation (comment) in public key. Your Ionic email address will be used',
                        aliases: ['C'],
                        groups: [cli_framework_1.OptionGroup.Advanced],
                    },
                ],
            };
        });
    }
    preRun(inputs, options) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            yield this.checkForOpenSSH();
            this.env.session.getUserToken();
            if (!options['annotation']) {
                options['annotation'] = this.env.config.get('user.email');
            }
            cli_framework_1.validate(String(options['type']), 'type', [cli_framework_1.contains(SSH_KEY_TYPES, { caseSensitive: false })]);
        });
    }
    run(inputs, options) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { getGeneratedPrivateKeyPath } = yield Promise.resolve().then(() => require('../../lib/ssh'));
            const { bits, annotation } = options;
            const keyPath = inputs[0] ? format_1.expandPath(String(inputs[0])) : yield getGeneratedPrivateKeyPath(this.env.config.get('user.id'));
            const keyPathDir = path.dirname(keyPath);
            const pubkeyPath = `${keyPath}.pub`;
            if (!(yield utils_fs_1.pathExists(keyPathDir))) {
                yield utils_fs_1.mkdirp(keyPathDir, 0o700); // tslint:disable-line
                this.env.log.msg(`Created ${chalk_1.default.bold(format_1.prettyPath(keyPathDir))} directory for you.`);
            }
            if (yield utils_fs_1.pathExists(keyPath)) {
                const confirm = yield this.env.prompt({
                    type: 'confirm',
                    name: 'confirm',
                    message: `Key ${chalk_1.default.bold(format_1.prettyPath(keyPath))} exists. Overwrite?`,
                });
                if (confirm) {
                    yield utils_fs_1.unlink(keyPath);
                }
                else {
                    this.env.log.msg(`Not overwriting ${chalk_1.default.bold(format_1.prettyPath(keyPath))}.`);
                    return;
                }
            }
            this.env.log.info('Enter a passphrase for your private key.\n' +
                `You will be prompted to provide a ${chalk_1.default.bold('passphrase')}, which is used to protect your private key should you lose it. (If someone has your private key, they can impersonate you!) Passphrases are recommended, but not required.\n`);
            const shellOptions = { stdio: 'inherit', showCommand: false, showError: false };
            yield this.env.shell.run('ssh-keygen', ['-q', '-t', String(options['type']), '-b', String(bits), '-C', String(annotation), '-f', keyPath], shellOptions);
            this.env.log.nl();
            this.env.log.rawmsg(`Private Key (${chalk_1.default.bold(format_1.prettyPath(keyPath))}): Keep this safe!\n` +
                `Public Key (${chalk_1.default.bold(format_1.prettyPath(pubkeyPath))}): Give this to all your friends!\n\n`);
            this.env.log.ok('A new pair of SSH keys has been generated!');
            this.env.log.nl();
            this.env.log.msg(`${chalk_1.default.bold('Next steps:')}\n` +
                ` * Add your public key to Ionic: ${chalk_1.default.green('ionic ssh add ' + format_1.prettyPath(pubkeyPath))}\n` +
                ` * Use your private key for secure communication with Ionic: ${chalk_1.default.green('ionic ssh use ' + format_1.prettyPath(keyPath))}`);
        });
    }
}
exports.SSHGenerateCommand = SSHGenerateCommand;
