"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const format_1 = require("@ionic/cli-framework/utils/format");
const chalk_1 = require("chalk");
const base_1 = require("./base");
class SSHListCommand extends base_1.SSHBaseCommand {
    getMetadata() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return {
                name: 'list',
                type: 'global',
                summary: 'List your SSH public keys on Ionic',
                options: [
                    {
                        name: 'json',
                        summary: 'Output SSH keys in JSON',
                        type: Boolean,
                    },
                ],
            };
        });
    }
    preRun() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            yield this.checkForOpenSSH();
        });
    }
    run(inputs, options) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { SSHKeyClient } = yield Promise.resolve().then(() => require('../../lib/ssh'));
            const { findHostSection, getConfigPath, isDirective, loadFromPath } = yield Promise.resolve().then(() => require('../../lib/ssh-config'));
            const { json } = options;
            const user = this.env.session.getUser();
            const token = this.env.session.getUserToken();
            const sshkeyClient = new SSHKeyClient({ client: this.env.client, user, token });
            const paginator = sshkeyClient.paginate();
            const [r] = paginator;
            const res = yield r;
            if (json) {
                process.stdout.write(JSON.stringify(res.data));
            }
            else {
                let activeFingerprint;
                let foundActiveKey = false;
                const sshConfigPath = getConfigPath();
                const conf = yield loadFromPath(sshConfigPath);
                const section = findHostSection(conf, this.env.config.getGitHost());
                if (section && section.config) {
                    const [identityFile] = section.config.filter(line => {
                        return isDirective(line) && line.param === 'IdentityFile';
                    });
                    if (isDirective(identityFile)) {
                        const output = yield this.env.shell.output('ssh-keygen', ['-E', 'sha256', '-lf', identityFile.value], { fatalOnError: false });
                        activeFingerprint = output.trim().split(' ')[1];
                    }
                }
                if (res.data.length === 0) {
                    this.env.log.warn(`No SSH keys found. Use ${chalk_1.default.green('ionic ssh add')} to add keys to Ionic.`);
                    return;
                }
                const keysMatrix = res.data.map(sshkey => {
                    const data = [sshkey.fingerprint, sshkey.name, sshkey.annotation];
                    if (sshkey.fingerprint === activeFingerprint) {
                        foundActiveKey = true;
                        return data.map(v => chalk_1.default.bold(v));
                    }
                    return data;
                });
                const table = format_1.columnar(keysMatrix, {
                    headers: ['fingerprint', 'name', 'annotation'],
                });
                if (foundActiveKey) {
                    this.env.log.nl();
                    this.env.log.msg(`The row in ${chalk_1.default.bold('bold')} is the key that this computer is using. To change, use ${chalk_1.default.green('ionic ssh use')}.`);
                }
                this.env.log.nl();
                this.env.log.rawmsg(table);
                this.env.log.nl();
                this.env.log.ok(`Showing ${chalk_1.default.bold(String(res.data.length))} SSH key${res.data.length === 1 ? '' : 's'}.`);
                this.env.log.nl();
            }
        });
    }
}
exports.SSHListCommand = SSHListCommand;
